<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Helpers\CommonHelper;
use App\Models\IndicatorDataEntry;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class UtifIndicatorListingController extends Controller
{


    public function __construct()
    {
        $this->middleware('permission:View UTIF Indicator Listing', ['only' => ['index']]);
        $this->middleware('auth');
    }

    /**START
     * FUNCTION: index
     * DESCRIPTION: This function is used to fetch the data for UTIF Indicator Listing page.
     * In this function we are fetching the data for UTIF Indicator Listing page.
     * We are fetching the data for the following filters:
     * 1. Department 2. Goal 3. Target 4. Year 5. Indicator
     * Return the view with the fetched data.
     * Parameters: Request $request -> indicator, department, goal, target, year, 
     * Return: View -> admin_pages/utifIndicatorListing/index
     * -------------------------------------------------------------
     * Author: SUHEL KHAN 
     * Date: 17-May-2024
     * PMS TASK ID: 148718
     * CHANGES: FIRST WRITE UP
     * -------------------------------------------------------------
     */
    public function index(Request $request)
    {
        $departments = [];
        $targets = [];
        $indicators = [];

        $logData = array();
        //prepared the log file name and location
        $file_name = date('Y-m-d') . '_utif_indicator_listing.txt';
        $location = 'logs/info-log/' . auth()->user()->email . '/utif-indicator-listing';
        $logData['user'] = auth()->user()->email;
        $logData['file'] = __FILE__;
        $logData['method'] = __FUNCTION__;
        $logData['information'] = "User accessed the UTIF Indicator Listing page.";
        //using dataActionLogger function to log the information
        CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);

        //get all goals
        $goals = CommonHelper::getGoalData('all');
        if (!($goals)) {
            $goals = [];
        }

        //select district reporting_year from IndicatorDataEntry
        try {
            $indicatorYearsData = IndicatorDataEntry::select('reporting_year')->distinct()->get()->toArray();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');

            $logData['information'] = "SQL query exception" . $e->getMessage() . " Query: " . $e->getSql();
            //using errorLogger function to log the error
            CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
        }

        // get the reporting years in ascending order. Now the financial year will be shown like 2022-2023
        $indicatorYears = array_column($indicatorYearsData, 'reporting_year');
        

        /*
        select 
            `indicator_data_entries`.`id`,
            `indicator_data_entries`.`indicator_id`,
            `indicator_data_entries`.`indicator_number`,
            `indicator_data_entries`.`indicator_name`,
            `indicator_data_entries`.`department_name`,
            `indicator_data_entries`.`reporting_year`,
            `indicator_data_entries`.`target_value`,
            `indicator_data_entries`.`indicator_value`,
            `indicator_data_entries`.`reporter_name`,
            `indicator_data_entries`.`approver_name`, 
        CASE
            WHEN previous.indicator_value IS NULL THEN ''
            WHEN indicator_data_entries.indicator_value > previous.indicator_value THEN 'up'
            WHEN indicator_data_entries.indicator_value < previous.indicator_value THEN 'down'
            WHEN indicator_data_entries.indicator_value = previous.indicator_value THEN 'straight'
            ELSE ''
        END AS trend
        From 
            `indicator_data_entries` 
        Left join 
            `indicator_data_entries` as `previous` 
        On 
            `indicator_data_entries`.`indicator_id` = `previous`.`indicator_id` 
            and `previous`.`reporting_year` = '2021-2022' 
        Where
            `indicator_data_entries`.`reporting_year` = '2022-2023'
            and `indicator_data_entries`.`approver_id` is not null
            and `indicator_data_entries`.`department_id` is not null
            and `indicator_data_entries`.`unit` != 'Absolute' 
            and `indicator_data_entries`.`status` in ('Approved', 'Completed') 
        Llimit 
            50 offset 0
        */
        $query = IndicatorDataEntry::query();
        $query->select(
            'indicator_data_entries.id',
            'indicator_data_entries.indicator_id',
            'indicator_data_entries.indicator_number',
            'indicator_data_entries.indicator_name',
            'indicator_data_entries.department_name',
            'indicator_data_entries.reporting_year',
            'indicator_data_entries.target_value',
            'indicator_data_entries.indicator_value',
            'indicator_data_entries.reporter_name',
            'indicator_data_entries.approver_name',
            'indicator_data_entries.indicator_value_comment',
            DB::raw("
                CASE
                    WHEN previous.indicator_value IS NULL THEN ''
                    WHEN indicator_data_entries.indicator_value > previous.indicator_value THEN 'up'
                    WHEN indicator_data_entries.indicator_value < previous.indicator_value THEN 'down'
                    WHEN indicator_data_entries.indicator_value = previous.indicator_value THEN 'straight'
                    ELSE ''
                END AS trend
            ")
        );

        $user = auth()->user();
        if ($user->hasRole('Department-User') || $user->hasRole('Department-Approver') || $user->hasRole('Department-HOD')) {
            // if the user is Department-User or Department-Approver, then fetch the indicator_data_entries based on the user's department_id
            $query->where('indicator_data_entries.department_id', $user->department_id);

            //Fetch the departments based on the user's department_id
            $departments = CommonHelper::getDepartments($user->department_id);
            if (!($departments)) {
                $departments = [];
            }
        } else {
            // if the user is Super Admin or Admin, then fetch all the departments
            $departments = CommonHelper::getDepartments('all');
            if (!($departments)) {
                $departments = [];
            }

            if ($request->has('department') && !empty($request->input('department'))) {
                $query->where('indicator_data_entries.department_id', $request->input('department'));
            }
        }

        //Condition prepared to fetch the indicators for indicator filter based on the selected goal, target and type (dif or utif)
        if ($request->has('goal') && !empty($request->input('goal'))) {
            //Fetch targets according to the applied filter.
            $targets = CommonHelper::getTargets('all', $request->input('goal'));
            if (!($targets)) {
                $targets = [];
            }

            //to add the condition of goal id in the query
            $query->where('indicator_data_entries.goal_id', $request->input('goal'));
        } else {
            //fetch the all targets
            $targets = CommonHelper::getTargets('all');
            if (!($targets)) {
                $targets = [];
            }
        }

        if ($request->has('target') && !empty($request->input('target'))) {
            //to add the condition of target_id in the query.
            $query->where('indicator_data_entries.target_id', $request->input('target'));
        }
        $yearToShowData = '';
        if ($request->has('year') && !empty($request->input('year'))) {
            $reportingYears = '';
            //to add the condition of reporting year in the query.
            $yearToShowData = $request->input('year');
            $reportingYears = trim($request->input('year'));
            $query->where('indicator_data_entries.reporting_year', $reportingYears);
        } else {
            $reportingYears = [];
            //fetch the if from ReportingYear where is_default = 'Yes'
            $year = CommonHelper::getDefaultYear();
            if (!($year)) {
                $year = [];
            }
            if (isset($year) && !empty($year)) {
                $yearToShowData = $year;
                $reportingYears[] = $year . "-" . ($year + 1);
                $query->where('indicator_data_entries.reporting_year', $reportingYears);
            }
        }

        if($request->has('indicator') && !empty($request->input('indicator'))) {
            $query->where('indicator_data_entries.indicator_id', $request->input('indicator'));
        }

        $query->leftJoin('indicator_data_entries as previous', function ($join) use ($yearToShowData) {
            $join->on('indicator_data_entries.indicator_id', '=', 'previous.indicator_id')
                ->where('previous.reporting_year', '=', $yearToShowData);
        });
        $query->whereNotNull('indicator_data_entries.approver_id');
        $query->whereNotNull('indicator_data_entries.department_id');
        $query->where('indicator_data_entries.unit', '!=', 'Absolute');
        $query->whereIn('indicator_data_entries.status', ['Approved', 'Completed']);

        $paginatate = env('INDICATOR_PAGINATION_LIMIT', '50');

        try {
            $indicators = $query->paginate($paginatate);
            $sql = $query->toSql();

            // Get the bindings
            $bindings = $query->getBindings();

            // Replace the placeholders with the actual bindings
            foreach ($bindings as $binding) {
                $value = is_numeric($binding) ? $binding : "'$binding'";
                $sql = preg_replace('/\?/', $value, $sql, 1);
            }
            //log the query in log file
            $logData['information'] = "SQL query to fetch the UTIF Indicators: " . $sql;
            //using dataActionLogger function to log the information
            CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
        }
        return view('admin_pages.utifIndicatorListing.index', ['departments' => $departments, 'targets' => $targets, 'indicators' => $indicators, 'indicatoryears' => $indicatorYears, 'reportingYears' => $reportingYears, 'goals' => $goals,'filter_year' => $yearToShowData]);
    }
    /**END
     * FUNCTION: index
     */ 
}
