<?php

namespace App\Http\Controllers;

use App\Helpers\CommonHelper;
use Illuminate\Http\Request;
use App\Helpers\PublicPageHelper;
use App\Models\District;
use App\Models\ReportingYear;
use App\Models\SdgScoreCategorization;
use Illuminate\Support\Facades\DB;
use Illuminate\Database\QueryException;
use App\Jobs\ProcessEntriesJob;
use App\Jobs\NotifyCompletionJob;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use App\Models\IndicatorDataEntry;
use App\Models\Goal;

class PublicDashboardController extends Controller
{

    /* Start: 
     * Function Name: index
     * Display the public dashboard with district-wise SDG index data.
     * This method retrieves and prepares data to display the public dashboard, including district-wise
     * SDG (Sustainable Development Goals) index data for a specific goal and district. It accepts
     * optional parameters for the goal ID and district code to filter the data accordingly.
     * If no parameters are provided or the specified goal or district does not exist, it falls back to
     * default values.
     *
     * @param  string|null $goal        (Optional) The ID of the goal for filtering the data.
     * @param  string|null $districtCode (Optional) The code of the district for filtering the data.
     * @return \Illuminate\View\View    The view for the public dashboard with the prepared data.
    * -------------------------------------------------------------------------
    * AUTHOR: SUHEL KHAN 
    * DATE: 19-April-2024
    * PM ID: #148098
    * CHANGES: FIRST WRITE-UP
    * -------------------------------------------------------------------------
    */
    public function index($goal = null, $districtCode = null)
    {
        $district_indicator_best_performance = [];
        $district_indicator_require_attention = [];
        $district_indicator_require_immidiate_action = [];
        $goal_utif_indicator_data = [];
        $goal_dif_indicator_data = [];
        $division_district_goals = [];
        $division_district_goal_indicators = [];
        $sunrise_best_performance = [];
        $sunrise_require_attention = [];
        $sunrise_require_immidiate_action = [];
        $max_value = '';
        $indicator_district = '';
        $distflag = 1;
        $goalflag = 1;
        $default_district = [];
        $goaldata = [];
        $defaultYearPair = '';

        if ($goal != null && $districtCode != null) {
            try {
                $district = District::where('district_code', $districtCode)
                    ->select('id', 'district_name')
                    ->where('status', 'Active')
                    ->first();
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');

                $district = [];
            }
            if (isset($district) && !empty($district)) {
                $default_district['id'] = $district->id;
                $default_district['district_name'] = $district->district_name;
            } else {
                $distflag = 0;
            }

            $defaultGoalId = CommonHelper::getGoalData($goal);
            if (!($defaultGoalId)) {
                $defaultGoalId  = CommonHelper::getGoalData('1');
            }
        } else {
            $distflag = 0;
            $goalflag = 0;
        }
        //Added flag to check if district is not found(for map) then get the default district, By default we are showing data of default district on public dashboard.
        if ($distflag == 0) {
            try {
                $defaultDistrict = District::select('id', 'district_name')
                    ->where('is_default_district', '1')
                    ->where('status', 'Active')
                    ->first();
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');

                $defaultDistrict = [];
            }
            $default_district['id'] = $defaultDistrict->id;
            $default_district['district_name'] = $defaultDistrict->district_name;
        }
        if ($goalflag == 0) {
            $defaultGoalId  = CommonHelper::getGoalData('1');
            if (!($defaultGoalId)) {
                $defaultGoalId = [];
            }
        }
        try {
            $defaultYear = ReportingYear::where('is_default', 'Yes')->first('reporting_year');
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');

            $defaultYear = [];
        }
        $defaultYear = explode('-', $defaultYear->reporting_year);
        $defaultYear = $defaultYear[0];
        $defaultYearPair = ($defaultYear - 2) . '-' . substr($defaultYear, 2, 4);

        // to get last three years from default year
        $lastThreeYears = CommonHelper::getLastThreeYears($defaultYear);

        $reportingYears = CommonHelper::getReportingYears($lastThreeYears);
        // Fetch the score categories
        try {
            $scoreCategories = SdgScoreCategorization::all();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');

            $scoreCategories = [];
        }

        // to get all districts order by 
        $allDistricts = CommonHelper::getDistricts('all');
        if (!($allDistricts)) {
            $allDistricts = [];
        }


        $goaldata = CommonHelper::getGoalData('all');
        if (!($goaldata)) {
            $goaldata = [];
        }


        $singleGoaldata = "";

        // Get all district sdg index data, this data is for goal and district wise
        $all_district_sdg_index_data = PublicPageHelper::getDistrictSdgIndexes($lastThreeYears, $scoreCategories, 'All', 'All');
        if (!$all_district_sdg_index_data) {
            $all_district_sdg_index_data = [];
        }

        // To get the District SDG score, There is no dependency on the goal_id it's overall score for the district
        $all_district_index_data = PublicPageHelper::getDistrictIndexes($lastThreeYears, $scoreCategories, 'All');
        if (!$all_district_index_data) {
            $all_district_index_data = [];
        }

        // To get the District SDG score, There is no dependency on the goal_id it's overall score for the district
        $district_map_data = PublicPageHelper::getDistrictCodeWithScoreCategories($allDistricts, $all_district_index_data, $defaultYear);
        if (!$district_map_data) {
            $district_map_data = [];
        }

        // Task #157546 Creation of Home Page of the Dashboard.
        // Author : Almaaz Ahmed
        // Date : 17-01-2025
        // Using different condition the page is redirected to the map view page and ut indicator frameword page

        if(isset($goal) && isset($districtCode)){
            return view('public_dashboard.map-view', [
                'goaldata' => $goaldata, 'defaultDistrict' => $default_district, 'all_district_sdg_index_data' => $all_district_sdg_index_data, 'reportingYears' => $reportingYears,
                'all_district_index_data' => $all_district_index_data, 'allDistricts' => $allDistricts, 'defaultYear' => $defaultYear, 'default_goal' => $defaultGoalId, 'district_map_data' => $district_map_data,
                'district_indicator_require_immidiate_action' => $district_indicator_require_immidiate_action, 'district_indicator_require_attention' => $district_indicator_require_attention,
                'district_indicator_best_performance' => $district_indicator_best_performance, 'indicator_district' => $indicator_district, 'singleGoaldata' => $singleGoaldata,
                'goal_dif_indicator_data' => $goal_dif_indicator_data, 'goal_utif_indicator_data' => $goal_utif_indicator_data, 'division_district_goals' => $division_district_goals,
                'division_district_goal_indicators' => $division_district_goal_indicators, 'sunrise_best_performance' => $sunrise_best_performance, 'sunrise_require_attention' => $sunrise_require_attention,
                'sunrise_require_immidiate_action' => $sunrise_require_immidiate_action, 'datagoal' => $goaldata, 'max_value' => $max_value, 'defaultYearPair' => $defaultYearPair, 'year' => $defaultYear
            ]);
        }else if(request()->routeIs('dashboard')){
            return view('public_dashboard.ut_indicator_framework', [
                'goaldata' => $goaldata, 'defaultDistrict' => $default_district, 'all_district_sdg_index_data' => $all_district_sdg_index_data, 'reportingYears' => $reportingYears,
                'all_district_index_data' => $all_district_index_data, 'allDistricts' => $allDistricts, 'defaultYear' => $defaultYear, 'default_goal' => $defaultGoalId, 'district_map_data' => $district_map_data,
                'district_indicator_require_immidiate_action' => $district_indicator_require_immidiate_action, 'district_indicator_require_attention' => $district_indicator_require_attention,
                'district_indicator_best_performance' => $district_indicator_best_performance, 'indicator_district' => $indicator_district, 'singleGoaldata' => $singleGoaldata,
                'goal_dif_indicator_data' => $goal_dif_indicator_data, 'goal_utif_indicator_data' => $goal_utif_indicator_data, 'division_district_goals' => $division_district_goals,
                'division_district_goal_indicators' => $division_district_goal_indicators, 'sunrise_best_performance' => $sunrise_best_performance, 'sunrise_require_attention' => $sunrise_require_attention,
                'sunrise_require_immidiate_action' => $sunrise_require_immidiate_action, 'datagoal' => $goaldata, 'max_value' => $max_value, 'defaultYearPair' => $defaultYearPair, 'year' => $defaultYear
            ]);
        }else{
            return view('public_dashboard.dashboard', [
                'goaldata' => $goaldata, 'defaultDistrict' => $default_district, 'all_district_sdg_index_data' => $all_district_sdg_index_data, 'reportingYears' => $reportingYears,
                'all_district_index_data' => $all_district_index_data, 'allDistricts' => $allDistricts, 'defaultYear' => $defaultYear, 'default_goal' => $defaultGoalId, 'district_map_data' => $district_map_data,
                'district_indicator_require_immidiate_action' => $district_indicator_require_immidiate_action, 'district_indicator_require_attention' => $district_indicator_require_attention,
                'district_indicator_best_performance' => $district_indicator_best_performance, 'indicator_district' => $indicator_district, 'singleGoaldata' => $singleGoaldata,
                'goal_dif_indicator_data' => $goal_dif_indicator_data, 'goal_utif_indicator_data' => $goal_utif_indicator_data, 'division_district_goals' => $division_district_goals,
                'division_district_goal_indicators' => $division_district_goal_indicators, 'sunrise_best_performance' => $sunrise_best_performance, 'sunrise_require_attention' => $sunrise_require_attention,
                'sunrise_require_immidiate_action' => $sunrise_require_immidiate_action, 'datagoal' => $goaldata, 'max_value' => $max_value, 'defaultYearPair' => $defaultYearPair, 'year' => $defaultYear
            ]);
        }
    }
    /*End: index*/

    /* Start: 
     * Function Name: districtSdgScoreForGoalAndDefaultDistrict
     * Display the district SDG score for a specific goal and default district.
     * This method retrieves and prepares data to display the district SDG score for a specific goal and
     * the default district. It accepts the goal ID and district ID as input parameters and fetches the
     * corresponding data to display the score for the specified goal and district.
     *
     * @param  \Illuminate\Http\Request $request The HTTP request object containing the input data.
     * @return \Illuminate\View\View           The view for the district SDG score for the specified goal and district.
     * -------------------------------------------------------------------------
     * AUTHOR: SUHEL KHAN
     * DATE: 19-April-2024
     * PM ID: #148098
     * CHANGES: FIRST WRITE-UP
     * -------------------------------------------------------------------------
     */

    public function districtSdgScoreForGoalAndDefaultDistrict(Request $request)
    {
        try {
            $defaultYear = ReportingYear::where('is_default', 'Yes')->first('reporting_year');
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');

            return response()->view('index', ['message' => 'Something went wrong, we will be back in few minutes'], 500);
        }
        $defaultYear = explode('-', $defaultYear->reporting_year);
        $defaultYear = $defaultYear[0];

        // Fetch the score categories
        try {
            $scoreCategories = SdgScoreCategorization::all();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
        }

        // to get last three years from default year
        $lastThreeYears = CommonHelper::getLastThreeYears($defaultYear);
        $reportingYears = CommonHelper::getReportingYears($lastThreeYears);

        $goalId = $request->input('goal_id');
        $datagoal = CommonHelper::getGoalData($goalId);
        if (!($datagoal)) {
            $datagoal = [];
        }

        $districtId = $request->input('district_id');
        try {
            $default_district = District::where('id', $districtId)->first();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');

            $default_district = [];
        }

        $default_district['id'] = $districtId;

        $default_district_sdg_index_data = PublicPageHelper::getDistrictSdgIndexes($lastThreeYears, $scoreCategories, $goalId, $districtId);
        if (!$default_district_sdg_index_data) {
            $default_district_sdg_index_data = [];
        }
        return view('public_dashboard.goal_bise_score_for_default_district', ['all_district_sdg_index_data' => $default_district_sdg_index_data, 'default_goal' => $datagoal, 'defaultDistrict' => $default_district, 'reportingYears' => $reportingYears]);
    }
    /*End: districtSdgScoreForGoalAndDefaultDistrict*/

    /* Start: 
     * Function Name: district_indicators_for_year
     * Display the district indicators for a specific year.
     * This method retrieves and prepares data to display the district indicators for a specific year. It
     * accepts the district ID and year as input parameters and fetches the corresponding data to display
     * the indicators for the specified district and year.
     *
     * @param  \Illuminate\Http\Request $request The HTTP request object containing the input data.
     * @return \Illuminate\View\View           The view for the district indicators for the specified year.
     * -------------------------------------------------------------------------
     * AUTHOR: SUHEL KHAN
     * DATE: 19-April-2024
     * PM ID: #148098
     * CHANGES: FIRST WRITE-UP
     * -------------------------------------------------------------------------
     */
    /* Start: 
     * Function Name: district_indicators_for_year
     * Display the district indicators for a specific year.
     * This method retrieves and prepares data to display the district indicators for a specific year. It
     * accepts the district ID and year as input parameters and fetches the corresponding data to display
     * the indicators for the specified district and year.
     *
     * @param  \Illuminate\Http\Request $request The HTTP request object containing the input data.
     * @return \Illuminate\View\View           The view for the district indicators for the specified year.
     * -------------------------------------------------------------------------
     * AUTHOR: Almaaz Ahmed
     * DATE: 17-December-2024
     * PM ID: #156546
     * CHANGES: Need to send the district name and reporting years on the modal page. So that the user can see the district name and year on the modal page.
     * -------------------------------------------------------------------------
     */
    public function district_indicators_for_year(Request $request)
    {
        $sunriseGraphData['best_performance'] = [];
        $sunriseGraphData['require_attention'] = [];
        $sunriseGraphData['require_immidiate_action'] = [];
        $sunrise_best_performance = '';
        $sunrise_require_attention = '';
        $sunrise_require_immidiate_action = '';
        $district_id = $request->input('district_id');
        $year = $request->input('year');

        $district_indicator_best_performance = [];
        $district_indicator_require_attention = [];
        $district_indicator_require_immidiate_action = [];
        $indicator_district = '';
        $allDistricts = []; // to get all districts
        $reportingYears = [];

        if ($district_id != '') {
            try {
                // $allDistricts = CommonHelper::getDistricts('all'); // to get all districts

                //The changes is done so that the district dropdown can be populated region wise.
                $allDistricts = District::where('status', 'Active')
                    ->orderBy('region', 'ASC') // Sort by region
                    ->orderBy('district_name', 'ASC') // Sort alphabetically by district name
                    ->get();

                // to get the default year. The below code is used to get the previous three years.
                $defaultYear = ReportingYear::where('is_default', 'Yes')->first('reporting_year');
                $defaultYear = explode('-', $defaultYear->reporting_year);
                $defaultYear = $defaultYear[0];
                $lastThreeYears = CommonHelper::getLastThreeYears($defaultYear);
                $reportingYears = CommonHelper::getReportingYears($lastThreeYears); // to get the reporting years
                //EOC
                $district = District::where('id', $district_id)
                    ->select('id', 'district_name')
                    ->where('status', 'Active')
                    ->first();
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');

                $district = [];
            }
            if (isset($district) && !empty($district)) {
                $indicator_district = $district->district_name;
            }
        }

        /*-------------------- Sunrise Graph Data ----------------------*/
        // get all goals
        $datagoal = CommonHelper::getGoalData('all');
        if (!$datagoal) {
            $datagoal = [];
        }

        //to fetch the best performance, require attention and require immidiate action data for the district
        $sunriseGraphData = PublicPageHelper::getSunriseGraphData($district_id, $year, $datagoal);
        if (!$sunriseGraphData) {
            $sunriseGraphData = [];
        }

        // to simplify the data for the graph
        foreach ($datagoal as $goal) {
            if (isset($sunriseGraphData['best_performance']) && !empty($sunriseGraphData['best_performance'])) {
                if (!(array_key_exists($goal['goal_id'], $sunriseGraphData['best_performance']))) {
                    $sunriseGraphData['best_performance'][$goal['goal_id']] = 0;
                }
            }
            if (isset($sunriseGraphData['require_attention']) && !empty($sunriseGraphData['require_attention'])) {
                if (!(array_key_exists($goal['goal_id'], $sunriseGraphData['require_attention']))) {
                    $sunriseGraphData['require_attention'][$goal['goal_id']] = 0;
                }
            }
            if (isset($sunriseGraphData['require_immidiate_action']) && !empty($sunriseGraphData['require_immidiate_action'])) {
                if (!(array_key_exists($goal['goal_id'], $sunriseGraphData['require_immidiate_action']))) {
                    $sunriseGraphData['require_immidiate_action'][$goal['goal_id']] = 0;
                }
            }
        }
        // Sort the arrays by key 
        if (isset($sunriseGraphData['best_performance']) && !empty($sunriseGraphData['best_performance'])) {
            ksort($sunriseGraphData['best_performance']);
        }
        if (isset($sunriseGraphData['require_attention']) && !empty($sunriseGraphData['require_attention'])) {
            ksort($sunriseGraphData['require_attention']);
        }
        if (isset($sunriseGraphData['require_immidiate_action']) && !empty($sunriseGraphData['require_immidiate_action'])) {
            ksort($sunriseGraphData['require_immidiate_action']);
        }

        $sum_values = [];
        if (!empty($sunriseGraphData['best_performance'])) {
            foreach ($sunriseGraphData['best_performance'] as $key => $value) {
                $sum = 0;
                if (!empty($sunriseGraphData['best_performance'])) {
                    $sum = $sunriseGraphData['best_performance'][$key];
                }
                if (!empty($sunriseGraphData['require_attention'])) {
                    $sum += $sunriseGraphData['require_attention'][$key];
                }
                if (!empty($sunriseGraphData['require_immidiate_action'])) {
                    $sum += $sunriseGraphData['require_immidiate_action'][$key];
                }

                $sum_values[$key] = $sum;
            }
        }

        //$sunrise_max_value = max($sum_values);
        $sunrise_max_value = !empty($sum_values) ? max($sum_values) : 0;

        if (isset($sunriseGraphData['best_performance']) && !empty($sunriseGraphData['best_performance'])) {
            $sunrise_best_performance = implode(',', $sunriseGraphData['best_performance']);
        }

        if (isset($sunriseGraphData['require_attention']) && !empty($sunriseGraphData['require_attention'])) {
            $sunrise_require_attention = implode(',', $sunriseGraphData['require_attention']);
        }

        if (isset($sunriseGraphData['require_immidiate_action']) && !empty($sunriseGraphData['require_immidiate_action'])) {
            $sunrise_require_immidiate_action = implode(',', $sunriseGraphData['require_immidiate_action']);
        }


        /*-------------------- Sunrise Graph Data ----------------------*/

        $district_indicator_data = PublicPageHelper::getDistrictIndicators($district_id, $year);
        if (!$district_indicator_data) {
            $district_indicator_data = [];
        }

        if (!empty($district_indicator_data) && count($district_indicator_data) > 0) {
            if (isset($district_indicator_data['best_performance']) && count($district_indicator_data['best_performance']) > 0) {

                $district_indicator_best_performance = $district_indicator_data['best_performance'];
            }
            if (isset($district_indicator_data['require_attention']) && count($district_indicator_data['require_attention']) > 0) {
                $district_indicator_require_attention = $district_indicator_data['require_attention'];
            }
            if (isset($district_indicator_data['require_immidiate_action']) && count($district_indicator_data['require_immidiate_action']) > 0) {
                $district_indicator_require_immidiate_action = $district_indicator_data['require_immidiate_action'];
            }
        }
        //new chagnes required to passed alldistricts, reportingyears and selected year to the view
        return view(
            'public_dashboard.district_indicator_year_bise_data',
            [
                'best_permonace' => $district_indicator_best_performance, 'require_attention' => $district_indicator_require_attention,
                'require_immidiate_action' => $district_indicator_require_immidiate_action, 'indicator_district' => $indicator_district,
                'sunrise_best_performance' => $sunrise_best_performance, 'sunrise_require_attention' => $sunrise_require_attention, 'sunrise_require_immidiate_action' => $sunrise_require_immidiate_action,
                'datagoal' => $datagoal, 'max_value' => $sunrise_max_value, 'allDistricts' => $allDistricts, 'reportingYears' => $reportingYears, 'selectedYear' => $year
            ]
        );
    }
    /*End: district_indicators_for_year*/

    /* Start: 
     * Function Name: sunrise_graph_data
     * Display the goal indicator data for a select bar graph on the sunrise page.
     * This method retrieves and prepares data to display the goal indicator data for a select bar graph on the sunrise page. It
     *
     * @param  \Illuminate\Http\Request $request The HTTP request object containing the input data.
     * @return \Illuminate\View\View           The view for the goal indicator data for the specified year.
     * -------------------------------------------------------------------------
     * AUTHOR: Almaaz Ahmed
     * DATE: 17-Dec-2024
     * Task ID: #156381 Implement a click event on the sunrise graph to show respective indicators on the screen.
     * -------------------------------------------------------------------------
     */
    public function sunrise_graph_data(Request $request)
    {
        $districtId = $request->district_id;
        $year = $request->year;
        $goalName = $request->goal_name;
        $category = $request->category;
        $goalId = $request->goal_id;

        // select `indicator_id`, `indicator_number`, `indicator_name`, `goal_name`, `goal_id`, `reporting_year`,
        // `district_id`, `district_name`, `indicator_normalised_value`, `indicator_value` from `indicator_data_entries`
        // where `district_id` = 7 and `reporting_year` = 2022-2023 and `goal_id` = 1 and `status` = Approved and unit != Absolute
        // and `indicator_normalised_value` < 64;

        // Start the query
        $query = IndicatorDataEntry::query();
        $query->select(
            'indicator_id',
            'indicator_number',
            'indicator_name',
            'goal_name',
            'goal_id',
            'reporting_year',
            'district_id',
            'district_name',
            'indicator_normalised_value',
            'indicator_value'
        );

        // Add base conditions
        if ($districtId) {
            $query->where('district_id', $districtId);
        }
        if ($year) {
            $query->where('reporting_year', $year);
        }
        if ($goalName) {
            $query->where('goal_id', $goalId);
        }
        $query->where('status', 'Approved')->WhereNot('unit', 'Absolute');

        // Apply category filtering
        if ($category === 'Best Performers') {
            $query->whereBetween('indicator_normalised_value', [74, 100]);
        } elseif ($category === 'Requires Attention') {
            $query->whereBetween('indicator_normalised_value', [64, 73]);
        } elseif ($category === 'Requires Immediate Action') {
            $query->where(function ($q) {
                $q->where('indicator_normalised_value', '<', 64)
                  ->orWhereNull('indicator_normalised_value');
            });
        }

        // Fetch the results
        try{
            $results = $query->get();
            // $results = $query->toSql();
            // $query_With_Bindings = $query->getBindings();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            $results = [];
        }

        // Return the results as a JSON response
        return response()->json([
            'data' => $results,
            'category' => $category,
            'message' => 'Data fetched successfully'
        ]);
    }
    /*End: sunrise_graph_data*/

    /* Start: 
     * Function Name: show_comparison_sdg_india
     * This function is used to load the comparison page
     *
     * @param  \Illuminate\Http\Request $request The HTTP request object containing the input data.
     * @return \Illuminate\View\View           The view for the goal indicator data for the specified year.
     * -------------------------------------------------------------------------
     * AUTHOR: Almaaz Ahmed
     * DATE: 18-December-2024
     * PM ID: #156544 The Comparison on SDG India Index section should display a detailed comparison of SDG scores with other states.
     * -------------------------------------------------------------------------
     */
    public function show_comparison_sdg_india(){
        $district_indicator_best_performance = [];
        $district_indicator_require_attention = [];
        $district_indicator_require_immidiate_action = [];
        $indicator_district = [];
        $sunrise_best_performance = [];
        $sunrise_require_attention = [];
        $sunrise_require_immidiate_action = [];
        $datagoal = [];
        $max_value = [];
        $allDistricts = [];
        $reportingYears = [];
        $year = [];
        $allGoals = [];
        try{
            $allGoals = Goal::select('id','goal_name')->where('status', 'Active')->get();
        }catch(QueryException $e){
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            $allGoals = [];
        }
        return view('public_dashboard.comparison_sdg_india',compact('district_indicator_best_performance', 'district_indicator_require_attention', 'district_indicator_require_immidiate_action', 'indicator_district', 'sunrise_best_performance', 'sunrise_require_attention', 'sunrise_require_immidiate_action', 'datagoal', 'max_value', 'allDistricts', 'reportingYears', 'year','allGoals'));
    }


    /* Start: 
     * Function Name: goal_indicator_data
     * Display the goal indicator data for a specific year.
     * This method retrieves and prepares data to display the goal indicator data for a specific year. It
     * accepts the goal ID and year as input parameters and fetches the corresponding data to display the
     * indicators for the specified goal and year.
     *
     * @param  \Illuminate\Http\Request $request The HTTP request object containing the input data.
     * @return \Illuminate\View\View           The view for the goal indicator data for the specified year.
     * -------------------------------------------------------------------------
     * AUTHOR: SUHEL KHAN
     * DATE: 19-April-2024
     * PM ID: #148098
     * CHANGES: FIRST WRITE-UP
     * -------------------------------------------------------------------------
     */
    public function goal_indicator_data(Request $request)
    {
        $goal_utif_indicator_data = [];
        $goal_dif_indicator_data = [];
        $single_years = [];
        $pairYears = [];
        $selectedPair = '';
        $utif_years = [];
        $goal_id = $request->input('goal_id');
        $year = $request->input('year');
        $area = $request->input('area'); // input from the public dashboard tab for redering different modals.

        try {
            $defaultYear = ReportingYear::where('is_default', 'Yes')->first('reporting_year');
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');

            $defaultYear = [];
        }
        $defaultYear = explode('-', $defaultYear->reporting_year);
        $defaultYear = $defaultYear[0];

        $reportingYears = CommonHelper::getReoprtingYears();
        if (!($reportingYears)) {
            $reportingYears = [];
        }
        if (isset($reportingYears) && !empty($reportingYears)) {
            $single_years = $reportingYears['singleYears'];
            $pairYears = $reportingYears['pairYears'];
        }

        try{
            $defaultYearBase = ReportingYear::where('is_default', 'Yes')->value('reporting_year'); // Fetch the the default year

            // Extract the base year from the default year (e.g., "2022-2023" -> 2022)
            $defaultYearBase = (int) explode('-', $defaultYear)[0];
    
            $financialyears = ReportingYear::where('status', 'Active')
                ->where('reporting_year_base', '<=', $defaultYearBase) // Include up to and including the default year
                ->orderBy('reporting_year_base', 'ASC') // Ensure chronological order
                ->select('id', 'reporting_year', 'reporting_year_base', 'is_default', 'status') // Specify columns
                ->get();
        }catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            $financialyears = [];
        }
        


        if (empty($request->input('utif_year'))) {
            // to get last three years from default year
            $lastThreeYears = CommonHelper::getLastThreeYears($defaultYear);
            foreach ($lastThreeYears as $yearPair) {
                $years = explode('-', $yearPair);
                $utif_years[] = (int)$years[0];
            }

            // Remove duplicate years
            $utif_years = array_unique($utif_years);

            // Re-index the array
            $utif_years = array_values($utif_years);

            $selectedPair = ($defaultYear - 2) . '-' . $defaultYear;
        } else {
            $selectedPair = $request->input('utif_year');
            $years = explode('-', $request->input('utif_year'));
            $startYear = (int) $years[0];
            $endYear = $startYear + 2;

            $yearPairs = [];
            for ($startYear; $startYear <= $endYear; $startYear++) {
                $yearPairs[] = $startYear . "-" . ($startYear + 1);
            }

            $lastThreeYears = $yearPairs;

            foreach ($lastThreeYears as $yearPair) {
                $years = explode('-', $yearPair);
                $utif_years[] = (int)$years[0];
            }

            // Remove duplicate years
            $utif_years = array_unique($utif_years);

            // Re-index the array
            $utif_years = array_values($utif_years);
        }

        // to get all districts
        if($area == 'district_index'){ // to get the district data for the specific goal and year if district is selected
            $allDistricts = CommonHelper::getDistricts('all');
            if (!($allDistricts)) {
                $allDistricts = [];
            }
        }
        

        $goaldata = CommonHelper::getGoalData($goal_id);
        if (!$goaldata) {
            $goaldata = [];
        }

        // To get the District SDG score for a specific goal and district [DIF]
        if($area == 'district_index'){ // to get the district data for the specific goal and year if district is selected
            $goal_dif_indicator_data = PublicPageHelper::getGoalDifIndicatorData($goal_id, $year);
            if (!$goal_dif_indicator_data) {
                $goal_dif_indicator_data = [];
            }
        }
        

        // To get the District SDG score for a specific goal [UTIF]
        $goal_utif_indicator_data = PublicPageHelper::getGoalIndicatorDataforLastThreeYears($goal_id, $lastThreeYears, 'utif');
        if (!$goal_utif_indicator_data) {
            $goal_utif_indicator_data = [];
        }

        if($area == 'district_index'){ // render the district modal if district is selected
            return view(
                'public_dashboard.goal_indicator_data_for_year',
                [
                    'goal_dif_indicator_data' => $goal_dif_indicator_data, 'goal_utif_indicator_data' => $goal_utif_indicator_data,
                    'reportingYears' => $reportingYears, 'allDistricts' => $allDistricts, 'singleGoaldata' => $goaldata, 'selected_year' => $year,
                    'single_years' => $single_years, 'utif_years' => $utif_years, 'financialyears' => $financialyears
                ]
            );
        }else{ // render the state modal if state is selected
            return view(
                'public_dashboard.goal_indicator_data_for_year_ut',
                [
                    'goal_utif_indicator_data' => $goal_utif_indicator_data,
                    'singleGoaldata' => $goaldata, 'selected_year' => $year,
                    'pairYears' => $pairYears, 'selectedPair' => $selectedPair, 'utif_years' => $utif_years, 'lastThreeYears' => $lastThreeYears
                ]
            );
        }
    }
    /*End: goal_indicator_data*/

    /* Start: 
     * Function Name: division_district_goals
     * Display the division district goals for a specific year.
     * This method retrieves and prepares data to display the division district goals for a specific year. It
     * accepts the district ID and year as input parameters and fetches the corresponding data to display the
     * goals for the specified district and year.
     *
     * @param  \Illuminate\Http\Request $request The HTTP request object containing the input data.
     * @return \Illuminate\View\View           The view for the division district goals for the specified year.
     * -------------------------------------------------------------------------
     * AUTHOR: SUHEL KHAN
     * DATE: 19-April-2024
     * PM ID: #148098
     * CHANGES: FIRST WRITE-UP
     * -------------------------------------------------------------------------
     */
    public function division_district_goals(Request $request)
    {
        $division_district_goals = [];

        $district_id = $request->input('district_id');

        try {
            $district_data = District::where('id', $district_id)->first();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');

            $district_data = [];
        }

        //The changes is done so that the financial year can shown
        $year = $request->input('year', date('Y')); // Get the input year or use the current year
        $nextYear = $year + 1; // Calculate the next year
        $yearRange = $year . '-' . $nextYear; // Concatenate the range

        try {
            $scoreCategories = SdgScoreCategorization::all();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');

            $scoreCategories = [];
        }

        $datagoal = CommonHelper::getGoalData('all');
        if (!$datagoal) {
            $datagoal = [];
        }

        // to get last three years from default year
        $lastThreeYears = CommonHelper::getLastThreeYears($year);

        $reportingYears = CommonHelper::getReportingYears($lastThreeYears);

        // To get the District SDG score for a specific goal and district
        $division_district_goals = PublicPageHelper::getDistrictSdgIndexes($lastThreeYears, $scoreCategories, 'All', $district_id);
        if (!$division_district_goals) {
            $division_district_goals = [];
        }

        // To get the District SDG score for a specific goal and district
        $composite_score = PublicPageHelper::getDistrictIndexes($lastThreeYears, $scoreCategories, $district_id);
        if (!$composite_score) {
            $composite_score = [];
        }
        return view('public_dashboard.division_district_goals', ['division_district_goals' => $division_district_goals, 'reportingYears' => $reportingYears, 'district_data' => $district_data, 'composite_score' => $composite_score, 'time_period' => $yearRange, 'datagoal' => $datagoal]);
    }
    /*End: division_district_goals*/

    /* Start: 
     * Function Name: division_district_goal_indicators
     * Display the division district goal indicators for a specific year.
     * This method retrieves and prepares data to display the division district goal indicators for a specific year. It
     * accepts the district ID and year as input parameters and fetches the corresponding data to display the
     * goal indicators for the specified district and year.
     *
     * @param  \Illuminate\Http\Request $request The HTTP request object containing the input data.
     * @return \Illuminate\View\View           The view for the division district goal indicators for the specified year.
     * -------------------------------------------------------------------------
     * AUTHOR: SUHEL KHAN
     * DATE: 19-April-2024
     * PM ID: #148098
     * CHANGES: FIRST WRITE-UP
     * -------------------------------------------------------------------------
     */
    public function division_district_goal_indicators(Request $request)
    {
        $division_district_goal_indicators = [];
        $pairYears = [];
        $dif_years = [];
        $selectedPair = '';
        $defaultYear = '';

        $district_id = $request->input('district_id');
        $goal_id = $request->input('goal_id');

        try {
            $district_data = District::where('id', $district_id)->first();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');

            $district_data = [];
        }

        $goal_data = CommonHelper::getGoalData($goal_id);
        if (!$goal_data) {
            $goal_data = [];
        }

        // to get default year
        try {
            $defaultYear = ReportingYear::where('is_default', 'Yes')->first('reporting_year');
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');

            $defaultYear = [];
        }
        $defaultYear = explode('-', $defaultYear->reporting_year);
        $defaultYear = $defaultYear[0];

        // to get last three years from default year
        $lastThreeYears = CommonHelper::getLastThreeYears($defaultYear);

        // to get all reporting years
        $reportingYears = CommonHelper::getReportingYears($lastThreeYears);

        $difreportingYears = CommonHelper::getReoprtingYears();
        if (!($reportingYears)) {
            $difreportingYears = [];
        }
        if (isset($difreportingYears) && !empty($difreportingYears)) {
            $pairYears = $difreportingYears['district_indicator_pair'];
        }

        if (empty($request->input('dif_year'))) {
            // to get last three years from default year
            // $diflastThreeYears = $lastThreeYears;
            // BOC: Updated the code to get the corrct financial year. Added by Almaaz Ahmed on 12-May-2025 PMS ID: 160301
            $diflastThreeYears = array_map(function ($range) {
                [$start, $end] = explode('-', $range);
                $newStart = (int)$start + 1;
                $newEnd = (int)$end + 1;
                return "{$newStart}-{$newEnd}";
            }, $lastThreeYears);
            // ECO: Updated the code to get the corrct financial year. Added by Almaaz Ahmed on 12-May-2025 PMS ID: 160301
            foreach ($lastThreeYears as $yearPair) {
                $years = explode('-', $yearPair);
                $dif_years[] = (int)$years[1];
            }
            // Remove duplicate years
            $dif_years = array_unique($dif_years);
            // Re-index the array
            $dif_years = array_values($dif_years);
            //updated the code to get the financial year in the format 2022-2023
            $selectedPair = ($defaultYear) . '-' . $defaultYear + 1;
        } else {
            $selectedPair = $request->input('dif_year');
            $years = explode('-', $request->input('dif_year'));
            $startYear = (int) $years[0];
            $defaultYear = $years[0];
            $endYear = $startYear + 1;
            $startYear = $startYear - 1;

            $yearPairs = [];
            for ($startYear; $startYear <= $endYear; $startYear++) {
                $yearPairs[] = $startYear . "-" . ($startYear + 1);
            }
            $diflastThreeYears = $yearPairs;
            foreach ($diflastThreeYears as $yearPair) {
                $years = explode('-', $yearPair);
                $dif_years[] = (int)$years[0];
            }
            // Remove duplicate years
            $dif_years = array_unique($dif_years);
            // Re-index the array
            $dif_years = array_values($dif_years);
        }

        // Fetch the score categories
        try {
            $scoreCategories = SdgScoreCategorization::all();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');

            $scoreCategories = [];
        }

        // To get the District SDG score for a specific goal and district
        $score = PublicPageHelper::getDistrictSdgIndexes($diflastThreeYears, $scoreCategories, $goal_id, $district_id);
        if (!$score) {
            $score = [];
        }

        // To get the District SDG score for a specific goal and district.
        $division_district_goal_indicators = PublicPageHelper::getGoalIndicatorDataforLastThreeYears($goal_id, $diflastThreeYears, 'dif', $district_id);
        if (!$division_district_goal_indicators) {
            $division_district_goal_indicators = [];
        }

        return view('public_dashboard.division_district_goal_indicators', [
            'division_district_goal_indicators' => $division_district_goal_indicators, 'reportingYears' => $reportingYears, 'district_data' => $district_data, 'goal_data' => $goal_data,
            'time_period' => $defaultYear, 'score' => $score, 'pairYears' => $pairYears, 'selectedPair' => $selectedPair, 'dif_years' => $dif_years, 'diflastThreeYears' => $diflastThreeYears
        ]);
    }
    /*End: division_district_goal_indicators*/

    public function call_procedures()
    {
        // Set maximum execution time of the script to 300 seconds (5 minutes)
        set_time_limit(500);
        // Increase the memory limit to 256M
        ini_set('memory_limit', '256M');

        $year = CommonHelper::getDefaultYear();
        if (!empty($year)) {
            $default_year = $year . '-' . ($year + 1);
        } else {
            $default_year = '';
        }
        try {
            $entries = DB::table('indicator_data_entries')
                ->join('indicators', 'indicators.id', '=', 'indicator_data_entries.indicator_id')
                ->select(
                    'indicator_data_entries.*',
                    'indicators.is_positive_indicator'
                )
                ->whereNotNull('indicator_data_entries.district_id')
                ->where('indicators.unit', '!=', 'Absolute')
                ->where('indicator_data_entries.reporting_year', $default_year)
                ->where('indicator_data_entries.status','Approved')
                ->get();
        } catch (QueryException $e) {
            $logData = [
                'file' => __FILE__,
                'method' => __FUNCTION__,
                'query' => $e->getSql(),
                'exception' => $e->getMessage()
            ];
            // Log exception
            Log::error("Error in {$logData['file']} file, {$logData['method']} method, Query: {$logData['query']}, Exception: {$logData['exception']} ");

            dd("Error in {$logData['file']} file, {$logData['method']} method, Query: {$logData['query']}, Exception: {$logData['exception']} ");
        }
        $i = 0;
        foreach ($entries as $entry) {
            $indicatorId = $entry->indicator_id;
            $districtId = $entry->district_id;
            $reportingYearId = $entry->reporting_year_id;
            $goal_id = $entry->goal_id;

            if ($entry->is_positive_indicator == 'Positive') {
                DB::statement("CALL calculateNormalizedValuePositiveIndicator('$indicatorId', '$districtId', '$reportingYearId')");
            } else {
                DB::statement("CALL calculateNormalizedValueNegativeIndicator('$indicatorId', '$districtId', '$reportingYearId')");
            }
            DB::statement("CALL calculateDistrictGoalScore('$goal_id', '$districtId', '$reportingYearId')");
            DB::statement("CALL calculateDistrictScore('$districtId', '$reportingYearId')");
            DB::statement("CALL setDistrictsRanks('$reportingYearId')");
            $i++;
        }
        return $i . ' Procedures called successfully';
    }
}
