<?php

namespace App\Http\Controllers;

use App\Helpers\CommonHelper;
use Illuminate\Http\Request;
use App\Models\IndicatorDataEntry;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\Log;
use App\Models\ReportingYear;
use App\Models\Indicator;
use App\Models\IndicatorUpdateHistory;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\IndicatorDataEntries;
use App\Mail\DataSubmittedMail;
use Illuminate\Support\Facades\Mail;
use App\Mail\ApprovalNotificationMail;
use App\Mail\ApprovalRequestNotificationMail;
use App\Mail\FinalApprovalNotificationMail;
use App\Mail\DistrictApproverRejectionMail;
use App\Mail\DistrictHODRejectionMail;
use App\Mail\DepartmentApproverRejectionMail;
use App\Mail\DepartmentHODRejectionMail;
use App\Mail\UpperLevelRejectionMail;
use App\Mail\AdminRejectionMail;
use App\Models\User;


class IndicatorDataEntriesController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:View Indicator Data Entry', ['only' => ['index']]);
        $this->middleware('permission:Create Indicator Data Entry', ['only' => ['create', 'store']]);
        $this->middleware('permission:Update Indicator Data Entry', ['only' => ['update', 'edit']]);
        $this->middleware('permission:Delete Indicator Data Entry', ['only' => ['destroy']]);
        $this->middleware('permission:Export Indicator Data Entry', ['only' => ['exportIndicatorDataEntries']]);
        $this->middleware('auth');
    }

    function array_insert_after($key, array &$array, $new_key, $new_value)
    {
        if (array_key_exists($key, $array)) {
            $new = [];
            foreach ($array as $k => $value) {
                $new[$k] = $value;
                if ($k === $key) {
                    $new[$new_key] = $new_value;
                }
            }
            return $new;
        }
        return $array;
    }

    /* Start: 
    * Function Name: index
    * Description: This function is used to display the list of Indicator Data Entries.
    * 1. If the user is Super Admin or Admin, then fetch all the districts and departments.
    * 2. If the user is District-User or District-Approver, then fetch the districts based on the user's district.
    * 3. If the user is Department-User or Department-Approver, then fetch the departments based on the user's department.
    * 4. If the user has selected the goal, then fetch the targets based on the selected goal.
    * 5. This function also fetches the Indicator Data Entries based on the selected filters.
    * PARAMETERS: 
    * @param Request $request (Request object to get the input data from the user)
    * @return View (Return the view with the list of Indicator Data Entries)
    * -------------------------------------------------------------------------
    * AUTHOR: SUHEL KHAN 
    * DATE: 07-May-2024
    * PM ID: #148110
    * CHANGES: FIRST WRITE-UP
    * -------------------------------------------------------------------------
    */
    public function index(Request $request)
    {
        $no_data = 0;
        $districts = [];
        $departments = [];
        $targets = [];
        $indicators = [];
        $district_filter = 'yes';
        $department_filter = 'yes';
        $type_flag = 1;
        $indicator_type = '';

        $logData = array();
        //prepared the log file name and location
        $file_name = date('Y-m-d') . '_indicator_data_entries.txt';
        $location = 'logs/info-log/' . auth()->user()->email . '/Indicator_data_entries';
        $logData['user'] = auth()->user()->email;
        $logData['file'] = __FILE__;
        $logData['method'] = __FUNCTION__;
        $logData['information'] = "User accessed Indicator Data Entries page.";
        //using dataActionLogger function to log the information
        CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);

        $user = auth()->user();
        $query = IndicatorDataEntry::query();
        // Check the user role and fetch the districts and departments based on the user role
        if ($user->hasRole('Super Admin') || $user->hasRole('Admin')) {
            // if the user is Super Admin or Admin, then fetch all the districts and departments
            $districts = CommonHelper::getDistricts('all');
            if (!$districts) {
                $districts = [];
            }

            $departments = CommonHelper::getDepartments('all');
            if (!$departments) {
                $departments = [];
            }
            $indicator_type = 'utif';
        } elseif ($user->hasRole('District-User') || $user->hasRole('District-Approver') || $user->hasRole('District-HOD')) {
            // if the user is District-User or District-Approver, then fetch the districts based on the user's district
            $districts = CommonHelper::getDistricts($user->district_id);
            if ($districts) {
                $type = 'dif';
                $query->where('indicator_data_entries.district_id', $user->district_id);
            }
            $type_flag = 0;
            $indicator_type = 'dif';
        } elseif ($user->hasRole('Department-User') || $user->hasRole('Department-Approver') || $user->hasRole('Department-HOD')) {
            // if the user is Department-User or Department-Approver, then fetch the departments based on the user's department
            $departments = CommonHelper::getDepartments($user->department_id);
            if ($departments) {
                $type = 'utif';
                $query->where('indicator_data_entries.department_id', $user->department_id);
            }
            $type_flag = 0;
            $indicator_type = 'utif';
        }

        // Fetch the targets based on the selected goal
        if ($request->has('goal')) {
            $targets = CommonHelper::getTargets('all', $request->input('goal'));
            if (!$targets) {
                $targets = [];
            }
        } else {
            // Fetch all the targets
            $targets = CommonHelper::getTargets('all', '1');
            if (!$targets) {
                $targets = [];
            }
        }

        // Fetch all the goals
        $goals = CommonHelper::getGoalData('all');
        if (!$goals) {
            $goals = [];
        }

        // Fetch the Indicator Data Entries based on the selected filters

        if ($request->has('target')) {
            if (is_numeric($request->input('target'))) {
                $query->where('indicator_data_entries.target_id', $request->input('target'));
            }
        }
        if ($request->has('goal') && !empty($request->input('goal'))) {
            if (is_numeric($request->input('goal'))) {
                $query->where('indicator_data_entries.goal_id', $request->input('goal'));
                $default_goal = $request->input('goal');
            }
        } else {
            $default_goal = 1;
            $query->where('indicator_data_entries.goal_id', $default_goal);
        }
        if ($request->has('type') && !empty($request->input('type'))) {
            $type = $request->input('type');
            if ($request->input('type') == 'dif') {
                if ($request->has('district') && !empty($request->input('district'))) {
                    if (is_numeric($request->input('district'))) {
                        $query->join('districts', 'indicator_data_entries.district_id', '=', 'districts.id');
                        $query->where('districts.status', 'Active');
                        $query->where('indicator_data_entries.district_id', $request->input('district'));
                    } else {
                        $query->whereNotNull('indicator_data_entries.district_id');
                    }
                } else {
                    $query->whereNotNull('indicator_data_entries.district_id');
                }
                $indicator_type = 'dif';
            } else if ($request->input('type') == 'utif') {
                if ($request->has('department') && !empty($request->input('department'))) {
                    if (is_numeric($request->input('department'))) {
                        $query->join('departments', 'indicator_data_entries.department_id', '=', 'departments.id');
                        $query->where('departments.status', 'Active');
                        $query->where('indicator_data_entries.department_id', $request->input('department'));
                    } else {
                        $query->whereNotNull('indicator_data_entries.department_id');
                    }
                } else {
                    $query->whereNotNull('indicator_data_entries.department_id');
                }
                $indicator_type = 'utif';
            } else {
                $type = 'utif';
                $query->whereNotNull('indicator_data_entries.department_id');
                $indicator_type = 'utif';
            }
        } else {
            if ($type_flag == 1) {
                $type = 'utif';
                $query->whereNotNull('indicator_data_entries.department_id');
                $indicator_type = 'utif';
            }
        }
        if ($request->has('status') && !empty($request->input('status'))) {
            $query->where('indicator_data_entries.status', $request->input('status'));
        }
        try {
            $defaultYear = ReportingYear::where('is_default', 'Yes')->first('id');
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            $defaultYear = '';
        }
        /*
        select
            `indicator_data_entries`.`id` as `indicator_data_entries_id`,
            `indicator_data_entries`.`indicator_id`,
            `indicator_data_entries`.`indicator_number`,
            `indicator_data_entries`.`indicator_name`,
            `indicator_data_entries`.`target_name`,
            `indicator_data_entries`.`goal_name`,
            `indicator_data_entries`.`unit`,
            'indicator_data_entries.value_type',
            `indicator_data_entries`.`target_value`,
            `indicator_data_entries`.`indicator_value`,
            `indicator_data_entries`.`baseline_year_id`,
            `indicator_data_entries`.`baseline_year`,
            `indicator_data_entries`.`reporting_year_id`,
            `indicator_data_entries`.`reporting_year`,
            `indicator_data_entries`.`department_id`,
            `indicator_data_entries`.`department_name`,
            `indicator_data_entries`.`district_id`,
            `indicator_data_entries`.`district_name`,
            `indicator_data_entries`.`status`,
            `indicator_data_entries`.`update_count`,
            `ide2`.`indicator_value` as `baseline_value`,
            `indicator_data_entries`.`created_at`
            
        From
            `indicator_data_entries`
            Left join `indicator_data_entries` as `ide2` on `indicator_data_entries`.`indicator_id` = `ide2`.`indicator_id`
                and `ide2`.`reporting_year_id` = `indicator_data_entries`.`baseline_year_id`
                and `ide2`.`district_id` = `indicator_data_entries`.`district_id`
            Inner join `indicators` on `indicator_data_entries`.`indicator_id` = `indicators`.`id`
            Inner join `targets` on `indicator_data_entries`.`target_id` = `targets`.`target_id`
            Inner join `goals` on `indicator_data_entries`.`goal_id` = `goals`.`id`
            Inner join `reporting_years` on `indicator_data_entries`.`reporting_year_id` = `reporting_years`.`id`
        Where
            `indicator_data_entries`.`goal_id` = 1
            and `indicator_data_entries`.`district_id` is not null
            and `indicator_data_entries`.`reporting_year_id` = 7
            and `goals`.`status` = 'Active'
            and `targets`.`status` = 'Active'
            and `indicators`.`is_discontinue` = 'No'
            order by
            `indicator_data_entries`.`indicator_id` desc
        Limit
            100
        Offset
            0
        */
        // Fetch the Indicator Data Entries based on the selected filters
        if (isset($defaultYear) && !empty($defaultYear)) {
            $query->select(
                'indicator_data_entries.id as indicator_data_entries_id',
                'indicator_data_entries.indicator_id',
                'indicator_data_entries.indicator_number',
                'indicator_data_entries.indicator_name',
                'indicator_data_entries.target_name',
                'indicator_data_entries.goal_name',
                'indicator_data_entries.unit',
                'indicator_data_entries.value_type',
                'indicator_data_entries.target_value',
                'indicator_data_entries.indicator_value',
                'indicator_data_entries.baseline_year_id',
                'indicator_data_entries.baseline_year',
                'indicator_data_entries.reporting_year_id',
                'indicator_data_entries.reporting_year',
                'indicator_data_entries.department_id',
                'indicator_data_entries.department_name',
                'indicator_data_entries.district_id',
                'indicator_data_entries.district_name',
                'indicator_data_entries.status',
                'indicator_data_entries.disapproval_reason',
                'indicator_data_entries.update_count',
                'ide2.indicator_value as baseline_value',
                'indicator_data_entries.created_at'
            );
            $query->where('indicator_data_entries.reporting_year_id', $defaultYear->id);
            if (isset($district_id) && !empty($district_id)) {
                $query->where('indicator_data_entries.district_id', $district_id);
            } else if (isset($department_id) && !empty($department_id)) {
                $query->where('indicator_data_entries.department_id', $department_id);
            }
            if ($indicator_type == 'dif') {
                $query->leftJoin('indicator_data_entries as ide2', function ($join) {
                    $join->on('indicator_data_entries.indicator_id', '=', 'ide2.indicator_id')
                        ->whereColumn('ide2.reporting_year_id', 'indicator_data_entries.baseline_year_id')
                        ->whereColumn('ide2.district_id', 'indicator_data_entries.district_id');
                });
            } else {
                $query->leftJoin('indicator_data_entries as ide2', function ($join) {
                    $join->on('indicator_data_entries.indicator_id', '=', 'ide2.indicator_id')
                        ->whereColumn('ide2.reporting_year_id', 'indicator_data_entries.baseline_year_id')
                        ->whereColumn('ide2.department_id', 'indicator_data_entries.department_id');
                });
            }
            $query->join('indicators', 'indicator_data_entries.indicator_id', '=', 'indicators.id');
            $query->join('targets', 'indicator_data_entries.target_id', '=', 'targets.target_id');
            $query->join('goals', 'indicator_data_entries.goal_id', '=', 'goals.id');
            $query->join('reporting_years', 'indicator_data_entries.reporting_year_id', '=', 'reporting_years.id');
            $query->where('goals.status', 'Active');
            $query->where('targets.status', 'Active');

            $query->where('indicators.is_discontinue', 'No');
            $query->orderBy('indicator_data_entries.indicator_id', 'desc');
            try {
                // Fetch the Indicator Data Entries based on the selected filters and set the pagination limit from the .env file
                $indicators = $query->paginate(env('INDICATOR_DATA_ENTRIES_PAGINATION_LIMIT', 100));
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');

                $indicators = [];
            }
        } else {
            $indicators = [];
        }

        if ($indicators->isEmpty()) {
            $message = "No data found in Indicator Data Entries.";
            $message .= "Query: " . $query->toSql();
        } else {
            $message = "User accessed Indicator Data Entries page.";
            // Process each indicator and add relevant data to the message
            $message .= "Data fetched for the indicator-data-entries with following query.\n";
            $message .= "Query: " . $query->toSql();
        }
        $logData['information'] = $message;
        CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
        return view('admin_pages.indicatorDataEntries.index', ['indicators' => $indicators, 'districts' => $districts, 'departments' => $departments, 'targets' => $targets, 'goals' => $goals, 'district_filter' => $district_filter, 'department_filter' => $department_filter, 'no_data' => $no_data, 'default_goal' => $default_goal, 'type' => $type]);
    }
    /*END - index(): Function to display the list of Indicator Data Entries*/

    /*Start:
    * Function Name: updateIndicatorValue
    * Description: This function is used to update the Indicator value in the indicators_data_entries table.
    * 1. Here we are updating the indicator value with multiple validations, both client and server side.
        a. Check if the Indicator value is greater than the maximum value of the Indicator.
        b. Check if the Indicator value is less than the minimum value of the Indicator.
        c. Check if the Indicator value is already updated.
        d. Check if the Indicator value is already approved.
        e. Check if the Indicator value is already updated 3 times.
    * 2. If the Indicator value is updated successfully, then update the Indicator Update History table.
    * PARAMETERS:
    * @param Request $request (Request object to get the input data from the user)
    * @return JSON (Response with the status of the Indicator value update)
    * -------------------------------------------------------------------------
    * AUTHOR: SUHEL KHAN
    * DATE: 08-May-2024
    * PM ID: #148110
    * CHANGES: FIRST WRITE-UP
    * -------------------------------------------------------------------------
    */
    public function updateIndicatorValue(Request $request)
    {
        $districtUser_sendNotification = false;
        $departmentUser_sendNotification = false;
        $districtApprover = false;
        $departmentApprover = false;
        $districtHOD = false;
        $departmentHOD = false;
        $approved_at_district = false;
        $approved_at_department = false;
        $approved_at_district_hod = false;
        $approved_at_department_hod = false;
        $confirmOverride = $request->input('confirm_override', false);

        // Initialize the variables
        $no_update_history = 0;
        $response = array();
        $file_name = date('Y-m-d') . '_indicator_value_update.txt';
        $location = 'logs/info-log/' . auth()->user()->email . '/Indicator_data_entries';
        $logData['user'] = auth()->user()->email;
        $logData['file'] = __FILE__;
        $logData['method'] = __FUNCTION__;

        $indicator_data_entries_id = $request->input('indicator_data_entry_id');
        $indicatorValue = $request->input('indicator_value');

        if ($indicatorValue == '' || $indicatorValue == null) {
            $response['status'] = 'error';
            $response['message'] = 'Indicator value is required.';
            $logData['information'] = "Indicator value is required.\n Indicator_data_entries ID: " . $indicator_data_entries_id . "\n Indicator Value: " . $indicatorValue;
            CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
            return response()->json($response);
        }
        // Check if the Indicator value is numeric
        if (!(is_numeric($indicatorValue))) {
            $response['status'] = 'error';
            $response['message'] = 'Indicator value should be numeric.';
            $logData['information'] = "Indicator value should be numeric.\n Indicator_data_entries ID: " . $indicator_data_entries_id . "\n Indicator Value: " . $indicatorValue;
            CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
            return response()->json($response);
        }

        //---------------------------------To find the indicator data entry and update the indicator value ---------------------------------//
        try {
            $indicatorDataEntry = IndicatorDataEntry::find($indicator_data_entries_id);
        } catch (\Error $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['error'] = $e->getMessage();
            // Log error to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'error');
            $response['input_disable'] = 'no';
            $response['status'] = 'error';
            $response['message'] = 'System encounter error while updating the Indicator value.';
            $logData['information'] = "System encounter error while updating the Indicator value.\n Error: " . $e->getMessage();
            CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
        }

        // Check if the Indicator value is already updated or approved or completed then disable the input box and save button
        if (isset($indicatorDataEntry) && !empty($indicatorDataEntry)) {

            if (isset($indicatorDataEntry->indicator_value) && $indicatorDataEntry->indicator_value == $indicatorValue) {
                $response['status'] = 'error';
                $response['message'] = 'The same Indicator value is already updated.';
                $logData['information'] = "Indicator value is already updated.\n Indicator_data_entries ID: " . $indicator_data_entries_id . "\n Indicator Value: " . $indicatorValue;
                CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                return response()->json($response);
            }

            if(!$confirmOverride && isset($indicator_data_entries_id) && !empty($indicator_data_entries_id)){
                // RAW QUERY
                // SELECT id,
                //     indicator_id,
                //     indicator_number,
                //     indicator_value,
                //     CASE 
                //         WHEN ABS((indicator_value - 13) / indicator_value) * 100 > 20 THEN 'false'
                //         ELSE 'true'
                //     END AS is_within_20_percent_variance
                // FROM indicator_data_entries
                // WHERE indicator_value IS NOT NULL
                // AND district_id = 1
                // AND indicator_id = 571
                // AND status = 'Approved'
                // ORDER BY reporting_year_id ASC
                // LIMIT 1;
                $varianceQuery = IndicatorDataEntry::select(
                        'id',
                        'indicator_id',
                        'indicator_number',
                        'indicator_value',
                        DB::raw("
                            CASE 
                                WHEN ABS((indicator_value - {$indicatorValue}) / indicator_value) * 100 > 20 THEN 'false'
                                ELSE 'true'
                            END AS is_within_20_percent_variance
                        ")
                    )
                    ->where('indicator_id', $indicatorDataEntry->indicator_id)
                    ->whereNotNull('indicator_value');

                // Proper conditional check for district_id or department_id
                if (!empty($indicatorDataEntry->district_id)) {
                    $varianceQuery->where('district_id', $indicatorDataEntry->district_id);
                } elseif (!empty($indicatorDataEntry->department_id)) {
                    $varianceQuery->where('department_id', $indicatorDataEntry->department_id);
                }

                $varianceQuery->where('status', 'Approved')
                    ->where('id', '!=', $indicator_data_entries_id)
                    ->orderBy('reporting_year_id', 'desc')
                    ->limit(1);

                try {
                    $varianceResult = $varianceQuery->first();
                } catch (QueryException $e) {
                    $logData['file'] = __FILE__;
                    $logData['method'] = __FUNCTION__;
                    $logData['query'] = $e->getSql();
                    $logData['exception'] = $e->getMessage();
                    CommonHelper::dataActionLogger($logData, 'exception');
                    $varianceResult = null;
                }

                if(isset($varianceResult) && $varianceResult->count() > 0 && $varianceResult->is_within_20_percent_variance == 'false'){
                    $response['status'] = 'error';
                    $response['message'] = 'A variance of ±20% has been observed compared to the prior year. Please recheck and confirm.';
                    return response()->json($response);
                }
            }

            $user = auth()->user();
            // Check if the Indicator value is already updated 3 times then disable the input box and save button
            if ($user->hasRole('District-User') || $user->hasRole('Department-User') || $user->hasRole('District-Approver') || $user->hasRole('Department-Approver')) {

                //------------------- to find the indicator update history -------------------//
                if (((strtolower($indicatorDataEntry->status) == 'approved' || strtolower($indicatorDataEntry->status) == 'completed'))) {
                    if (isset($indicatorDataEntry->update_count) && !empty($indicatorDataEntry->update_count)) {
                        if ($indicatorDataEntry->update_count >= 3) {
                            $response['status'] = 'error';
                            $response['message'] = 'You have reached the maximum limit of updating the Indicator value.';
                            $response['input_disable'] = 'yes';
                            $logData['information'] = "You have reached the maximum limit of updating the Indicator value.\n Indicator_data_entries ID: " . $indicator_data_entries_id . "\n Indicator Value: " . $indicatorValue;
                            CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                            return response()->json($response);
                        } else {
                            $indicatorDataEntry->update_count += 1;
                        }
                    } else {
                        $indicatorDataEntry->update_count = 1;
                    }
                }
            }
            //--------------------------------------------------------------------------------//
            try {
                $indicator = Indicator::find($indicatorDataEntry->indicator_id);
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');

                $response['input_disable'] = 'no';
                $response['status'] = 'error';
                $response['message'] = 'System encounter error while updating the Indicator value.';
                $logData['information'] = "System encounter error while updating the Indicator value.\n Error: " . $e->getMessage() . "\n Query: " . $e->getSql();
                CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
            }
            if (isset($indicator) && !empty($indicator)) {
                // Check if the Indicator value is greater than the maximum value of the Indicator
                if ($indicator->unit == 'Percentage' && ($indicatorValue > 100 || $indicatorValue < 0)) {
                    $response['status'] = 'error';
                    $response['message'] = 'Indicator value must be between 0 and 100 for Percentage unit';
                    $response['input_disable'] = 'no';
                    $logData['information'] = "Indicator value must be between 0 and 100 for Percentage unit\n Indicator_data_entries ID: " . $indicator_data_entries_id . "\n Indicator Value: " . $indicatorValue;
                    CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                    return response()->json($response);
                }

                $indicatorDataEntry->indicator_value = $indicatorValue;
                if ($user->hasRole('Admin') || $user->hasRole('Super Admin')) {
                    // $reponse['user'] = 'approver';
                    $indicatorDataEntry->data_added_by = $user->id;
                    $indicatorDataEntry->reporter_id = $user->id;
                    $indicatorDataEntry->reporter_name = $user->email;
                    
                    $indicatorDataEntry->level_one_approval_status = 'Approved';
                    $indicatorDataEntry->level_two_approval_status = 'Approved';

                    $indicatorDataEntry->level_three_approver_id = $user->id;
                    $indicatorDataEntry->level_three_approval_status = 'Approved';
                    $indicatorDataEntry->level_three_approval_date = now();
                    $indicatorDataEntry->status = 'Approved';
                    $indicatorDataEntry->approver_id = $user->id;
                    $indicatorDataEntry->approver_name = $user->email;
                    $indicatorDataEntry->approval_date = now();
                    $response['indicator_status'] = 'Approved';
                } else if ($user->hasRole('District-Approver')){
                    $indicatorDataEntry->data_added_by = $user->id;
                    $indicatorDataEntry->reporter_id = $user->id;
                    $indicatorDataEntry->reporter_name = $user->email;
                    $indicatorDataEntry->level_one_approver_id = $user->id;
                    $indicatorDataEntry->level_one_approval_status = 'Approved';
                    $indicatorDataEntry->level_one_approval_date = now();
                    $indicatorDataEntry->status = 'Sent_for_hod_approval';
                    $response['indicator_status'] = 'Sent for HOD approval';
                    $approved_at_district = true;
                } else if ($user->hasRole('Department-Approver')){
                    $indicatorDataEntry->data_added_by = $user->id;
                    $indicatorDataEntry->reporter_id = $user->id;
                    $indicatorDataEntry->reporter_name = $user->email;
                    $indicatorDataEntry->level_one_approver_id = $user->id;
                    $indicatorDataEntry->level_one_approval_status = 'Approved';
                    $indicatorDataEntry->level_one_approval_date = now();
                    $indicatorDataEntry->status = 'Sent_for_hod_approval';
                    $response['indicator_status'] = 'Sent for HOD approval';
                    $approved_at_department = true;
                } else if ($user->hasRole('District-HOD')){
                    $indicatorDataEntry->data_added_by = $user->id;
                    $indicatorDataEntry->reporter_id = $user->id;
                    $indicatorDataEntry->reporter_name = $user->email;
                    $indicatorDataEntry->level_two_approver_id = $user->id;
                    $indicatorDataEntry->level_two_approval_status = 'Approved';
                    $indicatorDataEntry->level_two_approval_date = now();
                    if($indicatorDataEntry->level_one_approver_id == null || $indicatorDataEntry->level_one_approver_id == ''){
                        $indicatorDataEntry->level_one_approver_id = $user->id;
                        $indicatorDataEntry->level_one_approval_status = 'Approved';
                        $indicatorDataEntry->level_one_approval_date = now();
                    }
                    $indicatorDataEntry->status = 'Sent_for_admin_approval';
                    $response['indicator_status'] = 'Sent for Admin Approval';
                    $approved_at_district_hod = true;
                } else if ($user->hasRole('Department-HOD')){
                    $indicatorDataEntry->data_added_by = $user->id;
                    $indicatorDataEntry->reporter_id = $user->id;
                    $indicatorDataEntry->reporter_name = $user->email;
                    $indicatorDataEntry->level_two_approver_id = $user->id;
                    $indicatorDataEntry->level_two_approval_status = 'Approved';
                    $indicatorDataEntry->level_two_approval_date = now();
                    if($indicatorDataEntry->level_one_approver_id == null || $indicatorDataEntry->level_one_approver_id == ''){
                        $indicatorDataEntry->level_one_approver_id = $user->id;
                        $indicatorDataEntry->level_one_approval_status = 'Approved';
                        $indicatorDataEntry->level_one_approval_date = now();
                    }
                    $indicatorDataEntry->status = 'Sent_for_admin_approval';
                    $response['indicator_status'] = 'Sent for Admin Approval';
                    $approved_at_department_hod = true;
                } else {
                    $response['user'] = 'normal_user';
                    $indicatorDataEntry->status = 'Sent_for_level_one_approval';
                    $indicatorDataEntry->data_added_by = $user->id;
                    $indicatorDataEntry->reporter_id = $user->id;
                    $indicatorDataEntry->reporter_name = $user->email;
                    $indicatorDataEntry->level_one_approver_id = null;
                    $indicatorDataEntry->level_one_approval_status = 'Pending';
                    $indicatorDataEntry->level_one_approval_date = null;
                    $indicatorDataEntry->level_two_approver_id = null;
                    $indicatorDataEntry->level_two_approval_status = 'Pending';
                    $indicatorDataEntry->level_two_approval_date = null;
                    $indicatorDataEntry->level_three_approver_id = null;
                    $indicatorDataEntry->level_three_approval_status = 'Pending';
                    $indicatorDataEntry->level_three_approval_date = null;
                    $response['indicator_status'] = 'Sent for level 1 approval';
                    if($user->hasRole('District-User')){
                        $districtUser_sendNotification = true;
                        $districtApprover_sendNotification = true;
                    }else if($user->hasRole('Department-User')){
                        $departmentUser_sendNotification = true;
                        $departmentApprover_sendNotification = true;
                    }
                }
                // Here we are checking if the user is Admin or Super Admin or District-Approver or Department-Approver then we will approve the indicator value while updating.
                try {
                    DB::beginTransaction();
                    if ($indicatorDataEntry->save()) {
                        $response['status'] = 'success';
                        $response['message'] = 'Indicator value updated successfully.';
                        $response['indicator_data_entries_id'] = $indicator_data_entries_id;
                        if ($user->hasRole('Super Admin') || $user->hasRole('Admin')) {
                            $response['input_disable'] = 'no';
                        } else {
                            $response['input_disable'] = 'yes';
                        }

                        $logData['information'] = "Indicator value updated successfully.\n Indicator_data_entries ID: " . $indicator_data_entries_id . "\n Indicator Value: " . $indicatorValue;
                        CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                        // If the Indicator value is updated successfully, then update the Indicator Update History table
                        DB::commit();
                        $district_id = $user->district_id;
                        $department_id = $user->department_id;
                        // Send mail to the approver
                        if($districtUser_sendNotification){
                            Mail::to($user->email_address)->send(new DataSubmittedMail($user));
                            if ($districtApprover_sendNotification) {
                                // get all the district approvers
                                $districtApprovers = commonHelper::getDistrictApprovers($district_id);
                                foreach ($districtApprovers as $districtApprover) {
                                    Mail::to($districtApprover->email_address)->send(new ApprovalRequestNotificationMail($user,$indicatorDataEntry,false,$districtApprover));
                                }
                            }
                        }else if($departmentUser_sendNotification){
                            Mail::to($user->email_address)->send(new DataSubmittedMail($user));
                            if ($departmentApprover_sendNotification) {
                                $department_id = $user->department_id;
                                // get all the department approvers
                                $departmentApprovers = commonHelper::getDepartmentApprovers($department_id);
                                foreach ($departmentApprovers as $departmentApprover) {
                                    Mail::to($departmentApprover->email_address)->send(new ApprovalRequestNotificationMail($user,$indicatorDataEntry,$departmentApprover, false));
                                }
                            }
                        }

                        if($approved_at_district){
                            // Send mail to the district approver hod
                            $district_approver_hods = CommonHelper::getDistrictHodApprover($district_id);
                            if($district_approver_hods != null){
                                foreach($district_approver_hods as $district_approver_hod){
                                    Mail::to($district_approver_hod->email_address)->send(new ApprovalRequestNotificationMail($user,$indicatorDataEntry,false,$district_approver_hod,false,'district'));
                                }
                            }
                        }else if($approved_at_department){
                            // Send mail to the department approver hod
                            $department_approver_hods = CommonHelper::getDepartmentHodApprover($department_id);
                            if($department_approver_hods != null){
                                foreach($department_approver_hods as $department_approver_hod){
                                    Mail::to($department_approver_hod->email_address)->send(new ApprovalRequestNotificationMail($user,$indicatorDataEntry,$department_approver_hod,false,false,'department'));
                                }
                            }
                        }else if($approved_at_district_hod || $approved_at_department_hod){
                            // Send mail to admin
                            $admins = CommonHelper::getAdmin();
                            if($admins != null){
                                foreach($admins as $admin){
                                    Mail::to($admin->email_address)->send(new ApprovalRequestNotificationMail($user,$indicatorDataEntry,false,false,$admin,'HOD'));
                                }
                            }
                        }
                    } else {
                        $response['status'] = 'error';
                        $response['message'] = 'System could not update the Indicator Value.';
                        $response['input_disable'] = 'no';
                    }
                } catch (QueryException $e) {
                    $logData['file'] = __FILE__;
                    $logData['method'] = __FUNCTION__;
                    $logData['query'] = $e->getSql();
                    $logData['exception'] = $e->getMessage();
                    // Log exception to file with file name and method information
                    CommonHelper::dataActionLogger($logData, 'exception');

                    $response['input_disable'] = 'no';
                    $response['status'] = 'error';
                    $response['message'] = 'System encounter error while updating the Indicator Value.';
                    $logData['information'] = "System encounter error while updating the Indicator Value.\n Error: " . $e->getMessage() . "\n Query: " . $e->getSql();
                    CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                    DB::rollBack();
                    return response()->json($response);
                }
            } else {
                $response['input_disable'] = 'no';
                $response['status'] = 'error';
                $response['message'] = 'System could not find this Indicator.';
                $logData['information'] = "System could not find this Indicator.\n Indicator_data_entries ID: " . $indicator_data_entries_id . "\n Indicator Value: " . $indicatorValue;
                CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
            }
        } else {
            $response['input_disable'] = 'no';
            $response['status'] = 'error';
            $response['message'] = 'System could not find this Indicator.';
            $logData['information'] = "System could not find this Indicator.\n Indicator_data_entries ID: " . $indicator_data_entries_id . "\n Indicator Value: " . $indicatorValue;
            CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
        }
        //--------------------------------------------------------------------------------//
        return response()->json($response);
    }
    /*END - updateIndicatorValue(): Function to update the Indicator value in the indicators_data_entries table*/

    /*Start:
    * Function Name: indicatorValueApproveAdisapprove
    * Description: This function is used to approve or disapprove the Indicator value in the indicators_data_entries table.
    * 1. Here we are approving or disapproving the indicator value with multiple validations, both client and server side.
        a. Check if the Indicator value is already approved or disapproved.
        b. Check if the Indicator value is already updated.
        c. Check if the Indicator value is already approved.
    * 2. If the Indicator value is approved or disapproved successfully, then update the Indicator Data Entries table.
    * PARAMETERS:
    * @param Request $request (Request object to get the input data from the user)
    * @return JSON (Response with the status of the Indicator value approve or disapprove)
    * -------------------------------------------------------------------------
    * Author: SUHEL KHAN
    * Date: 09-May-2024
    * PM ID: #148116
    * Changes: FIRST WRITE-UP
    * -------------------------------------------------------------------------
    * -------------------------------------------------------------------------
    * Author: Almaaz
    * Date: 12-May-2025
    * PM ID: #159987
    * Changes: SECOND WRITE-UP
    * 1. Added the approval disapproval flow email notification.
    * 2. Added the approval disapproval Flow for the HOD at the department and district level.
    * -------------------------------------------------------------------------
    */
    public function indicatorValueApproveAdisapprove(Request $request)
    {
        // Initialize the variables
        $response = array();
        $file_name = date('Y-m-d') . '_indicator_value_approve_disapprove.txt';
        $location = 'logs/info-log/' . auth()->user()->email . '/Indicator_data_entries';
        $logData['user'] = auth()->user()->email;
        $logData['file'] = __FILE__;
        $logData['method'] = __FUNCTION__;
        $user = auth()->user();
        $approval_level = '';
        $next_level = '';
        $department_id = $user->department_id; // Get the department ID of the logged in user
        $district_id = $user->district_id; // Get the district ID of the logged in user
        $approved_at_district = false;
        $approved_at_department = false;
        $approved_at_district_hod = false;
        $approved_at_department_hod = false;
        $final_approval = false;
        $disapproved_at_district = false;
        $disapproved_at_department = false;
        $disapproved_at_district_hod = false;
        $disapproved_at_department_hod = false;
        $disapproved_by_admin = false;
        // Check if the user is Admin or Super Admin or District-Approver or Department-Approver otherwise return the error message with status as unathorized access.
        if ($user->hasRole('Admin') || $user->hasRole('Super Admin') || $user->hasRole('District-Approver') || $user->hasRole('Department-Approver') || $user->hasRole('District-HOD') || $user->hasRole('Department-HOD')) {
            $indicator_data_entries_id = $request->input('indicator_data_entry_id');
            $disapproval_reason = $request->input('disapproval_reason');
            $approve_disapprove = $request->input('action');
            $indicator_value = $request->input('indicator_value');
            
            // BOC: Fetch the indicator data entry based on the district or department. The changes were done to prevent the user from approving the indicator data entry of other districts or departments. By ALMAAZ AHMED on 21-March-2025 PMS: 148137
            if($district_id != null || $district_id != ''){
                $indicatorDataEntry = IndicatorDataEntry::where('id', $indicator_data_entries_id)->where('district_id', $district_id)->first();
            }else if ( $department_id != null || $department_id != ''){
                $indicatorDataEntry = IndicatorDataEntry::where('id', $indicator_data_entries_id)->where('department_id', $department_id)->first();
            } else {
                $indicatorDataEntry = IndicatorDataEntry::find($indicator_data_entries_id);
            }

            // EOC: Fetch the indicator data entry based on the district or department. The changes were done to prevent the user from approving the indicator data entry of other districts or departments. By ALMAAZ AHMED on 21-March-2025 PMS: 148137
            if (isset($indicatorDataEntry) && !empty($indicatorDataEntry)) {
                // Check if the Indicator is already approved or disapproved
                if (strtolower($indicatorDataEntry->status) == 'approved' && strtolower($approve_disapprove) == 'approve') {
                    $response['status'] = 'error';
                    $response['message'] = 'This indicator is already ' . $indicatorDataEntry->status;
                    $response['indicator_status'] = $indicatorDataEntry->status;
                    $response['input_disable'] = 'no';
                    $logData['information'] = "Invalid action.\n Indicator_data_entries ID: " . $indicator_data_entries_id . "\n Indicator Status: " . $indicatorDataEntry->status;
                    CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                    return response()->json($response);
                } elseif (strtolower($indicatorDataEntry->status) == 'disapproved' && strtolower($approve_disapprove) == 'disapprove') {
                    $response['status'] = 'error';
                    $response['message'] = 'This indicator is already ' . $indicatorDataEntry->status;
                    $response['indicator_status'] = $indicatorDataEntry->status;
                    $response['input_disable'] = 'no';
                    $logData['information'] = "Invalid action.\n Indicator_data_entries ID: " . $indicator_data_entries_id . "\n Indicator Status: " . $indicatorDataEntry->status;
                    CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                    return response()->json($response);
                }

                // Check if the Indicator value is empty then return the error message
                if ($indicator_value == null || empty($indicator_value)) {
                    $response['status'] = 'error';
                    $response['message'] = 'Indicator value is required to approve or disapprove.';
                    $logData['information'] = "Indicator value is required to approve or disapprove.\n Indicator_data_entries ID: " . $indicator_data_entries_id;
                    CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                    return response()->json($response);
                }

                // Check if action is approve then update the status as Approved and if action is disapprove then update the status as Disapproved
                if ($approve_disapprove == 'approve') {
                    $user_role = $user->getRoleNames()->first();
                    if($user_role == 'District-Approver'){
                        // approved at district level
                        $indicatorDataEntry->level_one_approver_id = $user->id;
                        $indicatorDataEntry->level_one_approval_status = 'Approved';
                        $indicatorDataEntry->level_one_approval_date = now();
                        $indicatorDataEntry->status = 'Sent_for_hod_approval';
                        $approval_level = 'District';
                        $next_level = 'District-HOD';
                        $approved_at_district = true;
                        $response['indicator_status'] = 'Sent for HOD approval';
                    }else if($user_role == 'Department-Approver'){
                        $indicatorDataEntry->level_one_approver_id = $user->id;
                        $indicatorDataEntry->level_one_approval_status = 'Approved';
                        $indicatorDataEntry->level_one_approval_date = now();
                        $indicatorDataEntry->status = 'Sent_for_hod_approval';
                        $approval_level = 'Department';
                        $next_level = 'Department-HOD';
                        $approved_at_department = true;
                        $response['indicator_status'] = 'Sent for HOD approval';
                    }else if($user_role == 'District-HOD'){
                        $indicatorDataEntry->level_two_approver_id = $user->id;
                        $indicatorDataEntry->level_two_approval_status = 'Approved';
                        $indicatorDataEntry->level_two_approval_date = now();
                        $indicatorDataEntry->status = 'Sent_for_admin_approval';
                        if($indicatorDataEntry->level_one_approver_id == null || $indicatorDataEntry->level_one_approver_id == ''){
                            $indicatorDataEntry->level_one_approver_id = $user->id;
                            $indicatorDataEntry->level_one_approval_status = 'Approved';
                            $indicatorDataEntry->level_one_approval_date = now();
                        }
                        $approval_level = 'District-HOD';
                        $next_level = 'Admin';
                        $approved_at_district_hod = true;
                        $response['indicator_status'] = 'Sent for Admin Approval';
                    }else if($user_role == 'Department-HOD'){
                        $indicatorDataEntry->level_two_approver_id = $user->id;
                        $indicatorDataEntry->level_two_approval_status = 'Approved';
                        $indicatorDataEntry->level_two_approval_date = now();
                        $indicatorDataEntry->status = 'Sent_for_admin_approval';
                        if($indicatorDataEntry->level_one_approver_id == null || $indicatorDataEntry->level_one_approver_id == ''){
                            $indicatorDataEntry->level_one_approver_id = $user->id;
                            $indicatorDataEntry->level_one_approval_status = 'Approved';
                            $indicatorDataEntry->level_one_approval_date = now();
                        }
                        $approval_level = 'Department-HOD';
                        $next_level = 'Admin';
                        $approved_at_department_hod = true;
                        $response['indicator_status'] = 'Sent for Admin Approval';
                    }else if($user_role == 'Admin' || $user_role == 'Super Admin'){
                        $indicatorDataEntry->level_three_approver_id = $user->id;
                        $indicatorDataEntry->level_three_approval_status = 'Approved';
                        $indicatorDataEntry->level_three_approval_date = now();
                        if(($indicatorDataEntry->level_two_approver_id == null || $indicatorDataEntry->level_two_approver_id == '') && $indicatorDataEntry->department_id == null){
                            $indicatorDataEntry->level_two_approver_id = $user->id;
                            $indicatorDataEntry->level_two_approval_status = 'Approved';
                            $indicatorDataEntry->level_two_approval_date = now();
                        }
                        if(($indicatorDataEntry->level_one_approver_id == null || $indicatorDataEntry->level_one_approver_id == '') && $indicatorDataEntry->department_id == null){
                            $indicatorDataEntry->level_one_approver_id = $user->id;
                            $indicatorDataEntry->level_one_approval_status = 'Approved';
                            $indicatorDataEntry->level_one_approval_date = now();
                        }
                        if(($indicatorDataEntry->level_two_approver_id == null || $indicatorDataEntry->level_two_approver_id == '') && $indicatorDataEntry->district_id == null){
                            $indicatorDataEntry->level_two_approver_id = $user->id;
                            $indicatorDataEntry->level_two_approval_status = 'Approved';
                            $indicatorDataEntry->level_two_approval_date = now();
                        }
                        if(($indicatorDataEntry->level_one_approver_id == null || $indicatorDataEntry->level_one_approver_id == '') && $indicatorDataEntry->district_id == null){
                            $indicatorDataEntry->level_one_approver_id = $user->id;
                            $indicatorDataEntry->level_one_approval_status = 'Approved';
                            $indicatorDataEntry->level_one_approval_date = now();
                        }
                        $indicatorDataEntry->status = 'Approved';
                        $indicatorDataEntry->approval_date = now();
                        $response['indicator_status'] = 'Approved';
                        $final_approval = true;
                    }
                } else if ($approve_disapprove == 'disapprove') {
                    if (isset($disapproval_reason) && !empty($disapproval_reason)) {
                        $indicatorDataEntry->disapproval_reason = trim($disapproval_reason);
                        $disapproval_reason = $indicatorDataEntry->disapproval_reason;
                        $indicatorDataEntry->disapproval_date = now();
                    } else {
                        $response['status'] = 'error';
                        $response['message'] = 'Disapproval reason is required.';
                        $logData['information'] = "Disapproval reason is required.\n Indicator_data_entries ID: " . $indicator_data_entries_id;
                        CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                        return response()->json($response);
                    }

                    $user_role = $user->getRoleNames()->first();
                    if($user_role == 'District-Approver'){
                        // approved at district level
                        $indicatorDataEntry->level_one_approver_id = $user->id;
                        $indicatorDataEntry->level_one_approval_status = 'Rejected';
                        $indicatorDataEntry->level_one_approval_date = now();
                        $approval_level = 'District';
                        $disapproved_at_district = true;
                    }else if($user_role == 'Department-Approver'){
                        $indicatorDataEntry->level_one_approver_id = $user->id;
                        $indicatorDataEntry->level_one_approval_status = 'Rejected';
                        $indicatorDataEntry->level_one_approval_date = now();
                        $approval_level = 'Department';
                        $disapproved_at_department = true;
                    }else if($user_role == 'District-HOD'){
                        $indicatorDataEntry->level_two_approver_id = $user->id;
                        $indicatorDataEntry->level_two_approval_status = 'Rejected';
                        $indicatorDataEntry->level_two_approval_date = now();
                        if($indicatorDataEntry->level_one_approver_id == null || $indicatorDataEntry->level_one_approver_id == ''){
                            $indicatorDataEntry->level_one_approver_id = $user->id;
                            $indicatorDataEntry->level_one_approval_status = 'Rejected';
                            $indicatorDataEntry->level_one_approval_date = now();
                        }
                        $approval_level = 'District-HOD';
                        $disapproved_at_district_hod = true;
                    }else if($user_role == 'Department-HOD'){
                        $indicatorDataEntry->level_two_approver_id = $user->id;
                        $indicatorDataEntry->level_two_approval_status = 'Rejected';
                        $indicatorDataEntry->level_two_approval_date = now();
                        if($indicatorDataEntry->level_one_approver_id == null || $indicatorDataEntry->level_one_approver_id == ''){
                            $indicatorDataEntry->level_one_approver_id = $user->id;
                            $indicatorDataEntry->level_one_approval_status = 'Rejected';
                            $indicatorDataEntry->level_one_approval_date = now();
                        }
                        $approval_level = 'Department-HOD';
                        $disapproved_at_department_hod = true;
                    }else if($user_role == 'Admin' || $user_role == 'Super Admin'){
                        $indicatorDataEntry->level_three_approver_id = $user->id;
                        $indicatorDataEntry->level_three_approval_status = 'Rejected';
                        $indicatorDataEntry->level_three_approval_date = now();
                        if(($indicatorDataEntry->level_two_approver_id == null || $indicatorDataEntry->level_two_approver_id == '') && $indicatorDataEntry->department_id == null){
                            $indicatorDataEntry->level_two_approver_id = $user->id;
                            $indicatorDataEntry->level_two_approval_status = 'Rejected';
                            $indicatorDataEntry->level_two_approval_date = now();
                        }
                        if(($indicatorDataEntry->level_one_approver_id == null || $indicatorDataEntry->level_one_approver_id == '') && $indicatorDataEntry->department_id == null){
                            $indicatorDataEntry->level_one_approver_id = $user->id;
                            $indicatorDataEntry->level_one_approval_status = 'Rejected';
                            $indicatorDataEntry->level_one_approval_date = now();
                        }
                        if(($indicatorDataEntry->level_two_approver_id == null || $indicatorDataEntry->level_two_approver_id == '') && $indicatorDataEntry->district_id == null){
                            $indicatorDataEntry->level_two_approver_id = $user->id;
                            $indicatorDataEntry->level_two_approval_status = 'Rejected';
                            $indicatorDataEntry->level_two_approval_date = now();
                        }
                        if(($indicatorDataEntry->level_one_approver_id == null || $indicatorDataEntry->level_one_approver_id == '') && $indicatorDataEntry->district_id == null){
                            $indicatorDataEntry->level_one_approver_id = $user->id;
                            $indicatorDataEntry->level_one_approval_status = 'Rejected';
                            $indicatorDataEntry->level_one_approval_date = now();
                        }
                        $disapproved_by_admin = true;
                    }
                    $indicatorDataEntry->status = 'Disapproved';
                    $response['indicator_status'] = 'Disapproved';
                    $response['disapproval_reason'] = $disapproval_reason;
                } else {
                    $response['status'] = 'error';
                    $response['message'] = 'This action is not allowed, Either the data is already approved or disapproved.';
                    $logData['information'] = "Invalid action.\n Indicator_data_entries ID: " . $indicator_data_entries_id . "\n Indicator Status: " . $indicatorDataEntry->status;
                    CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                    return response()->json($response);
                }
                $indicatorDataEntry->approver_id = $user->id;
                $indicatorDataEntry->indicator_value = $indicator_value;
                $indicatorDataEntry->approver_name = $user->email;
                try {
                    if ($indicatorDataEntry->save()) {
                        $response['status'] = 'success';
                        $response['message'] = 'Indicator value ' . $approve_disapprove . 'd successfully.';
                        $response['indicator_data_entries_id'] = $indicator_data_entries_id;
                        // Check if the Indicator value is already updated 3 times then disable the input box and save button
                        if ($user->hasRole('Super Admin') || $user->hasRole('Admin')) {
                            $response['input_disable'] = 'no';
                        } else {
                            $response['input_disable'] = 'yes';
                        }
                        $logData['information'] = "Indicator value " . $approve_disapprove . "d successfully.\n Indicator_data_entries ID: " . $indicator_data_entries_id;
                        CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                        if($approved_at_district || $approved_at_department || $approved_at_district_hod || $approved_at_department_hod){
                            // Send mail to the data added by user
                            try{
                                $data_added_by_user = User::find($indicatorDataEntry->data_added_by);
                                if($data_added_by_user != null){
                                    Mail::to($data_added_by_user->email_address)->send(new ApprovalNotificationMail($data_added_by_user, $approval_level, $next_level, $indicatorDataEntry));
                                }
                            }catch (QueryException $e) {
                                $logData['file'] = __FILE__;
                                $logData['method'] = __FUNCTION__;
                                $logData['query'] = $e->getSql();
                                $logData['exception'] = $e->getMessage();
                                // Log exception to file with file name and method information
                                CommonHelper::dataActionLogger($logData, 'exception');
            
                                $response['status'] = 'error';
                                $response['message'] = 'System encounter error while ' . $approve_disapprove . ' the Indicator Value.';
                                $logData['information'] = "System encounter error while " . $approve_disapprove . " the Indicator Value.\n Error: " . $e->getMessage() . "\n Query: " . $e->getSql();
                                CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                                return response()->json($response);
                            }
                            

                            if($approved_at_district){
                                // Send mail to the district approver hod
                                $district_approver_hods = CommonHelper::getDistrictHodApprover($district_id);
                                if($district_approver_hods != null){
                                    foreach($district_approver_hods as $district_approver_hod){
                                        Mail::to($district_approver_hod->email_address)->send(new ApprovalRequestNotificationMail($user,$indicatorDataEntry,false,$district_approver_hod,false,'district'));
                                    }
                                }
                            }else if($approved_at_department){
                                // Send mail to the department approver hod
                                $department_approver_hods = CommonHelper::getDepartmentHodApprover($department_id);
                                if($department_approver_hods != null){
                                    foreach($department_approver_hods as $department_approver_hod){
                                        Mail::to($department_approver_hod->email_address)->send(new ApprovalRequestNotificationMail($user,$indicatorDataEntry,$department_approver_hod,false,false,'department'));
                                    }
                                }
                            }else if($approved_at_district_hod || $approved_at_department_hod){
                                // Send mail to admin
                                $admins = CommonHelper::getAdmin();
                                if($admins != null){
                                    foreach($admins as $admin){
                                        Mail::to($admin->email_address)->send(new ApprovalRequestNotificationMail($user,$indicatorDataEntry,false,false,$admin,'HOD'));
                                    }
                                }
                            }
                        }

                        if($disapproved_at_district){
                            // Send mail to the data added by user
                            try{
                                $data_added_by_user = User::find($indicatorDataEntry->data_added_by);
                                if($data_added_by_user != null){
                                    Mail::to($data_added_by_user->email_address)->send(new DistrictApproverRejectionMail($user,$data_added_by_user,$indicatorDataEntry, $disapproval_reason));
                                }
                            }catch (QueryException $e) {
                                $logData['file'] = __FILE__;
                                $logData['method'] = __FUNCTION__;
                                $logData['query'] = $e->getSql();
                                $logData['exception'] = $e->getMessage();
                                // Log exception to file with file name and method information
                                CommonHelper::dataActionLogger($logData, 'exception');
            
                                $response['status'] = 'error';
                                $response['message'] = 'System encounter error while ' . $approve_disapprove . ' the Indicator Value.';
                                $logData['information'] = "System encounter error while " . $approve_disapprove . " the Indicator Value.\n Error: " . $e->getMessage() . "\n Query: " . $e->getSql();
                                CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                                return response()->json($response);
                            }
                        }

                        if($disapproved_at_department){
                            // Send mail to the data added by user
                            try{
                                $data_added_by_user = User::find($indicatorDataEntry->data_added_by);
                                if($data_added_by_user != null){
                                    Mail::to($data_added_by_user->email_address)->send(new DepartmentApproverRejectionMail($user,$data_added_by_user,$indicatorDataEntry, $disapproval_reason));
                                }
                            }catch (QueryException $e) {
                                $logData['file'] = __FILE__;
                                $logData['method'] = __FUNCTION__;
                                $logData['query'] = $e->getSql();
                                $logData['exception'] = $e->getMessage();
                                // Log exception to file with file name and method information
                                CommonHelper::dataActionLogger($logData, 'exception');
            
                                $response['status'] = 'error';
                                $response['message'] = 'System encounter error while ' . $approve_disapprove . ' the Indicator Value.';
                                $logData['information'] = "System encounter error while " . $approve_disapprove . " the Indicator Value.\n Error: " . $e->getMessage() . "\n Query: " . $e->getSql();
                                CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                                return response()->json($response);
                            }
                        }

                        if($disapproved_at_district_hod){
                            // Send mail to the data added by user
                            try{
                                $data_added_by_user = User::find($indicatorDataEntry->data_added_by);
                                if($data_added_by_user != null){
                                    Mail::to($data_added_by_user->email_address)->send(new DistrictHODRejectionMail($user,$data_added_by_user,$indicatorDataEntry, $disapproval_reason));
                                }
                            }catch (QueryException $e) {
                                $logData['file'] = __FILE__;
                                $logData['method'] = __FUNCTION__;
                                $logData['query'] = $e->getSql();
                                $logData['exception'] = $e->getMessage();
                                // Log exception to file with file name and method information
                                CommonHelper::dataActionLogger($logData, 'exception');
            
                                $response['status'] = 'error';
                                $response['message'] = 'System encounter error while ' . $approve_disapprove . ' the Indicator Value.';
                                $logData['information'] = "System encounter error while " . $approve_disapprove . " the Indicator Value.\n Error: " . $e->getMessage() . "\n Query: " . $e->getSql();
                                CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                                return response()->json($response);
                            }

                            $districtApprovers = commonHelper::getDistrictApprovers($district_id);
                            foreach ($districtApprovers as $districtApprover) {
                                Mail::to($districtApprover->email_address)->send(new UpperLevelRejectionMail($districtApprover,$indicatorDataEntry,'District HOD', $disapproval_reason, $user));
                            }
                        }

                        if($disapproved_at_department_hod){
                            // Send mail to the data added by user
                            try{
                                $data_added_by_user = User::find($indicatorDataEntry->data_added_by);
                                if($data_added_by_user != null){
                                    Mail::to($data_added_by_user->email_address)->send(new DepartmentHODRejectionMail($user,$data_added_by_user,$indicatorDataEntry, $disapproval_reason));
                                }
                            }catch (QueryException $e) {
                                $logData['file'] = __FILE__;
                                $logData['method'] = __FUNCTION__;
                                $logData['query'] = $e->getSql();
                                $logData['exception'] = $e->getMessage();
                                // Log exception to file with file name and method information
                                CommonHelper::dataActionLogger($logData, 'exception');
            
                                $response['status'] = 'error';
                                $response['message'] = 'System encounter error while ' . $approve_disapprove . ' the Indicator Value.';
                                $logData['information'] = "System encounter error while " . $approve_disapprove . " the Indicator Value.\n Error: " . $e->getMessage() . "\n Query: " . $e->getSql();
                                CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                                return response()->json($response);
                            }

                            $departmentApprovers = commonHelper::getDepartmentApprovers($department_id);
                            foreach ($departmentApprovers as $departmentApprover) {
                                Mail::to($departmentApprover->email_address)->send(new UpperLevelRejectionMail($departmentApprover,$indicatorDataEntry,'Department HOD', $disapproval_reason, $user));
                            }
                        }

                        if($disapproved_by_admin){
                            // Send mail to the data added by user
                            try{
                                $data_added_by_user = User::find($indicatorDataEntry->data_added_by);
                                if($data_added_by_user != null){
                                    Mail::to($data_added_by_user->email_address)->send(new AdminRejectionMail($user,$data_added_by_user,$indicatorDataEntry, $disapproval_reason));
                                }
                            }catch (QueryException $e) {
                                $logData['file'] = __FILE__;
                                $logData['method'] = __FUNCTION__;
                                $logData['query'] = $e->getSql();
                                $logData['exception'] = $e->getMessage();
                                // Log exception to file with file name and method information
                                CommonHelper::dataActionLogger($logData, 'exception');
            
                                $response['status'] = 'error';
                                $response['message'] = 'System encounter error while ' . $approve_disapprove . ' the Indicator Value.';
                                $logData['information'] = "System encounter error while " . $approve_disapprove . " the Indicator Value.\n Error: " . $e->getMessage() . "\n Query: " . $e->getSql();
                                CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                                return response()->json($response);
                            }

                            if($indicatorDataEntry->district_id != null || $indicatorDataEntry->district_id != ''){
                                $district_approver_hods = CommonHelper::getDistrictHodApprover($indicatorDataEntry->district_id);
                                if($district_approver_hods != null){
                                    foreach($district_approver_hods as $district_approver_hod){
                                        Mail::to($district_approver_hod->email_address)->send(new UpperLevelRejectionMail($district_approver_hod,$indicatorDataEntry,'Admin', $disapproval_reason, $user));
                                    }
                                }

                                $districtApprovers = commonHelper::getDistrictApprovers($indicatorDataEntry->district_id);
                                if($districtApprovers != null){
                                    foreach ($districtApprovers as $districtApprover) {
                                        Mail::to($districtApprover->email_address)->send(new UpperLevelRejectionMail($districtApprover,$indicatorDataEntry,'District HOD', $disapproval_reason, $user));
                                    }
                                }
                            }else{
                                $department_approver_hods = CommonHelper::getDepartmentHodApprover($indicatorDataEntry->department_id);
                                if($department_approver_hods != null){
                                    foreach($department_approver_hods as $department_approver_hod){
                                        Mail::to($department_approver_hod->email_address)->send(new UpperLevelRejectionMail($department_approver_hod,$indicatorDataEntry,'Admin', $disapproval_reason, $user));
                                    }
                                }

                                $departmentApprovers = commonHelper::getDepartmentApprovers($indicatorDataEntry->department_id);
                                if($departmentApprovers != null){
                                    foreach ($departmentApprovers as $departmentApprover) {
                                        Mail::to($departmentApprover->email_address)->send(new UpperLevelRejectionMail($departmentApprover,$indicatorDataEntry,'Department HOD', $disapproval_reason, $user));
                                    }
                                }
                            }
                        }

                        if($final_approval){
                            // Send mail to the data added by user
                            try{
                                $data_added_by_user = User::find($indicatorDataEntry->data_added_by);
                                if($data_added_by_user != null){
                                    Mail::to($data_added_by_user->email_address)->send(new FinalApprovalNotificationMail($data_added_by_user, $user, $indicatorDataEntry));
                                }
                            }catch (QueryException $e) {
                                $logData['file'] = __FILE__;
                                $logData['method'] = __FUNCTION__;
                                $logData['query'] = $e->getSql();
                                $logData['exception'] = $e->getMessage();
                                // Log exception to file with file name and method information
                                CommonHelper::dataActionLogger($logData, 'exception');
            
                                $response['status'] = 'error';
                                $response['message'] = 'System encounter error while ' . $approve_disapprove . ' the Indicator Value.';
                                $logData['information'] = "System encounter error while " . $approve_disapprove . " the Indicator Value.\n Error: " . $e->getMessage() . "\n Query: " . $e->getSql();
                                CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                                return response()->json($response);
                            }
                        }
                        return response()->json($response);
                    } else {
                        $response['status'] = 'error';
                        $response['message'] = 'System could not ' . $approve_disapprove . ' the Indicator Value.';
                        $logData['information'] = "System could not " . $approve_disapprove . " the Indicator Value.\n Indicator_data_entries ID: " . $indicator_data_entries_id;
                        CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                    }
                } catch (QueryException $e) {
                    $logData['file'] = __FILE__;
                    $logData['method'] = __FUNCTION__;
                    $logData['query'] = $e->getSql();
                    $logData['exception'] = $e->getMessage();
                    // Log exception to file with file name and method information
                    CommonHelper::dataActionLogger($logData, 'exception');

                    $response['status'] = 'error';
                    $response['message'] = 'System encounter error while ' . $approve_disapprove . ' the Indicator Value.';
                    $logData['information'] = "System encounter error while " . $approve_disapprove . " the Indicator Value.\n Error: " . $e->getMessage() . "\n Query: " . $e->getSql();
                    CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                    return response()->json($response);
                }
            } else {
                $response['status'] = 'error';
                $response['message'] = 'System could not find this Indicator.';
                $logData['information'] = "System could not find this Indicator.\n Indicator_data_entries ID: " . $indicator_data_entries_id . "\n or User cannot perform the certain action";
                CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                return response()->json($response);
            }
        } else {
            $response['status'] = 'error';
            $response['message'] = 'You are not authorized to perform this action.';
            $logData['information'] = "You are not authorized to perform this action.\n User: " . $user->email . "\n Role: " . $user->role;
            CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
            return response()->json($response);
        }
    }
    /*END - indicatorValueApproveAdisapprove(): Function to approve or disapprove the Indicator value in the indicators_data_entries table*/

    /*Start:
    * Function Name: exportIndicatorDataEntries
    * Description: This function is used to export the Indicator Data Entries in the excel file.
    * -------------------------------------------------------------
    * PARAMETERS:
    * @param Request $request (Request object to get the input data from the user)
    * @return Excel (Return the excel file with the list of Indicator Data Entries)
    * -------------------------------------------------------------
    * Author: SUHEL KHAN
    * Date: 10-June-2024
    * PM ID: #148869
    * Changes: FIRST WRITE-UP
    * -------------------------------------------------------------
    */

    public function exportIndicatorDataEntries(Request $request)
    {
        $file_name = date('Y-m-d') . '_indicator_data_entries_export.txt';
        $location = 'logs/info-log/' . auth()->user()->email . '/Indicator_data_entries';
        $logData['user'] = auth()->user()->email;
        $logData['file'] = __FILE__;
        $logData['method'] = __FUNCTION__;
        $logData['information'] = "User exported the Indicator Data Entries.";
        CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);

        $user = auth()->user();
        $query = IndicatorDataEntry::query();
        // Check the user role and fetch the districts and departments based on the user role
        if ($user->hasRole('Super Admin') || $user->hasRole('Admin')) {
            // if the user is Super Admin or Admin, then fetch all the districts and departments
            $districts = CommonHelper::getDistricts('all');
            if (!$districts) {
                $districts = [];
            }

            $departments = CommonHelper::getDepartments('all');
            if (!$departments) {
                $departments = [];
            }
        } elseif ($user->hasRole('District-User') || $user->hasRole('District-Approver') || $user->hasRole('District-HOD')) {
            // if the user is District-User or District-Approver, then fetch the districts based on the user's district
            $districts = CommonHelper::getDistricts($user->district_id);
            if (!($districts)) {
                $districts = [];
            } else {
                $query->where('district_id', $user->district_id);
            }
        } elseif ($user->hasRole('Department-User') || $user->hasRole('Department-Approver') || $user->hasRole('Department-HOD')) {
            // if the user is Department-User or Department-Approver, then fetch the departments based on the user's department
            $departments = CommonHelper::getDepartments($user->department_id);
            if (!($departments)) {
                $departments = [];
            } else {
                $query->where('department_id', $user->department_id);
            }
        }

        // Fetch the targets based on the selected goal
        if ($request->has('goal')) {
            $targets = CommonHelper::getTargets('all', $request->input('goal'));
            if (!($targets)) {
                $targets = [];
            }
        } else {
            // Fetch all the targets
            $targets = CommonHelper::getTargets('all');
            if (!($targets)) {
                $targets = [];
            }
        }

        // Fetch all the goals
        $goals = CommonHelper::getGoalData('all');
        if (!($goals)) {
            $goals = [];
        }

        // Fetch the Indicator Data Entries based on the selected filters

        if ($request->has('target')) {
            if (is_numeric($request->input('target'))) {
                $query->where('indicator_data_entries.target_id', $request->input('target'));
            }
        }
        if ($request->has('goal') && !empty($request->input('goal'))) {
            if (is_numeric($request->input('goal'))) {
                $query->where('indicator_data_entries.goal_id', $request->input('goal'));
            }
        } else {
            $query->where('indicator_data_entries.goal_id', '1');
        }
        if ($request->has('type') && !empty($request->input('type'))) {
            if ($request->input('type') == 'dif') {
                if ($request->has('district') && !empty($request->input('district'))) {
                    if (is_numeric($request->input('district'))) {
                        $query->join('districts', 'indicator_data_entries.district_id', '=', 'districts.id');
                        $query->where('districts.status', 'Active');
                        $query->where('indicator_data_entries.district_id', $request->input('district'));
                    } else {
                        $query->whereNotNull('district_id');
                    }
                } else {
                    $query->whereNotNull('district_id');
                }
            } else if ($request->input('type') == 'utif') {
                if ($request->has('department') && !empty($request->input('department'))) {
                    if (is_numeric($request->input('department'))) {
                        $query->join('departments', 'indicator_data_entries.department_id', '=', 'departments.id');
                        $query->where('departments.status', 'Active');
                        $query->where('indicator_data_entries.department_id', $request->input('department'));
                    } else {
                        $query->whereNotNull('indicator_data_entries.department_id');
                    }
                } else {
                    $query->whereNotNull('indicator_data_entries.department_id');
                }
            }
        } else {
            $query->whereNotNull('indicator_data_entries.department_id');
        }
        try {
            $defaultYear = ReportingYear::where('is_default', 'Yes')->first('id');
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');

            $defaultYear = '';
        }
        /*
        select 
            `indicator_id`, 
            `indicator_number`, 
            `indicator_name`, 
            `target_name`, 
            `goal_name`,
            `unit`, 
            `target_value`, 
            `indicator_value`, 
            `baseline_year_id`, 
            `baseline_year`,
            `reporting_year_id`,
            `reporting_year`, 
            `department_id`, 
            `department_name`, 
            `district_id`, 
            `district_name`, 
            `created_at` 
        from  
            `indicator_data_entries`
        where  
            `reporting_year_id` = 7 and department_id is not null
        order by 
                `indicator_id` asc 
        limit 
                100 offset 0
        */
        // Fetch the Indicator Data Entries based on the selected filters
        if (isset($defaultYear) && !empty($defaultYear)) {
            $query->select('indicator_data_entries.id as indicator_data_entries_id', 'indicator_data_entries.indicator_id', 'indicator_data_entries.indicator_number', 'indicator_data_entries.indicator_name', 'indicator_data_entries.unit', 'indicator_data_entries.value_type', 'indicator_data_entries.baseline_year', 'indicator_data_entries.target_value', 'indicator_data_entries.indicator_value', 'indicator_data_entries.status');

            $query->where('indicator_data_entries.reporting_year_id', $defaultYear->id);
            // $query->where('indicator_data_entries.status', 'Pending');

            if (isset($district_id) && !empty($district_id)) {
                $query->where('indicator_data_entries.district_id', $district_id);
            } else if (isset($department_id) && !empty($department_id)) {
                $query->where('indicator_data_entries.department_id', $department_id);
            }

            $query->join('indicators', 'indicator_data_entries.indicator_id', '=', 'indicators.id');
            $query->join('targets', 'indicator_data_entries.target_id', '=', 'targets.target_id');
            $query->join('goals', 'indicator_data_entries.goal_id', '=', 'goals.id');
            $query->where('goals.status', 'Active');
            $query->where('targets.status', 'Active');

            $query->where('indicators.is_discontinue', 'No');
            $query->orderBy('indicator_id', 'asc');
            try {
                // Fetch the Indicator Data Entries based on the selected filters and set the pagination limit from the .env file
                $indicators = $query->get();

                if (!$indicators->isEmpty()) {
                    $baseLineDataQuery = Indicator::query();
                    $baseLineDataQuery->select('indicator_id', 'indicator_value');
                    $baseLineDataQuery->join('indicator_data_entries', function ($join) {
                        $join->on('indicators.id', '=', 'indicator_data_entries.indicator_id')
                            ->on('indicators.baseline_year_id', '=', 'indicator_data_entries.reporting_year_id');
                    });
                    $baseLineDataQuery->distinct();
                    $baselineValue = $baseLineDataQuery->get()->unique('indicator_id');
                    if (isset($baselineValue) && !empty($baselineValue)) {

                        //Code added to set the baseline value after the value_type, so that it will be displayed in the excel sheet
                        foreach ($indicators as $indicator) {
                            if (!empty($baselineValue)) {
                                // Convert the indicator model to an array
                                $indicatorArray = $indicator->getAttributes();

                                if ($baselineValue->has($indicator->indicator_id)) {
                                    $indicatorArray = $this->array_insert_after('value_type', $indicatorArray, 'baseline_value', $baselineValue[$indicator->indicator_id]->indicator_value);
                                } else {
                                    $indicatorArray = $this->array_insert_after('value_type', $indicatorArray, 'baseline_value', '');
                                }

                                // Convert the array back to a model
                                $indicator->setRawAttributes($indicatorArray);

                                // Make the baseline_value attribute visible
                                $indicator->makeVisible('baseline_value');
                            }
                        }

                        //Prepare the excel sheet 
                        $filename = 'Indicator_data_entries_' . now()->format('Y-m-d_H-i-s') . '.xlsx';
                        Excel::store(new IndicatorDataEntries($indicators), 'export/indicator_data_entries/' . $filename);

                        // Generate the full path to the saved file
                        $filePath = storage_path('app/export/indicator_data_entries/' . $filename);

                        return response()->download($filePath, $filename);
                    } else {
                        $indicators = [];
                    }
                } else {
                    $message = "No data found in Indicator Data Entries.";
                    $message .= "Query: " . $query->toSql();
                    $logData['information'] = $message;
                    CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                    return redirect()->back()->with('error', "No data found in Indicator Data Entries.");
                }
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');

                $indicators = [];
            }
        } else {
            $indicators = [];
        }
    }
}
