<?php

namespace App\Http\Controllers;

use App\Helpers\CommonHelper;
use App\Models\IndicatorDataEntry;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Database\QueryException;
use App\Models\ReportingYear;
use App\Models\Indicator;
use App\Exports\IndicatorDataEntryTemplate;
use Illuminate\Http\Request;
use PhpOffice\PhpSpreadsheet\IOFactory;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;
use App\Mail\ApprovalRequestNotificationMail;
use Illuminate\Support\Facades\Mail;

class ImportRawDataController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:View Import Data', ['only' => ['index']]);
        $this->middleware('auth');
    }

    public function index()
    {
        return view('admin_pages.import-raw-data.index', ['status' => '', 'downloadLink' => '']);
    }

    /**START: Function add the baseline value to the indicator data entries array
     * @param $key - key to insert the new key after
     * @param array $array - array to insert the new key and value
     * @param $new_key - new key to insert
     * @param $new_value - new value to insert
     * @return array - return the new array with the new key and value
     * -----------------------------------------------------------------------------------
     * @description - This function is used to insert the new key and value after the specified key in the array
     * -----------------------------------------------------------------------------------
     * Author: SUHEL KHAN
     * Date: 11-June-2024
     * PMS TASK ID: 148869
     * -----------------------------------------------------------------------------------
     */
    function array_insert_after($key, array &$array, $new_key, $new_value)
    {
        if (array_key_exists($key, $array)) {
            $new = [];
            foreach ($array as $k => $value) {
                $new[$k] = $value;
                if ($k === $key) {
                    $new[$new_key] = $new_value;
                }
            }
            return $new;
        }
        return $array;
    }
    /**END: Function add the baseline value to the indicator data entries array */


    /**START:
     * Function- downloadIndicatorsRawData()
     * @description - This function is used to download the Indicator Data Entries in the excel sheet.
     * Here we are fetching all those indicator data entries which are in pending status for year defalut.
     * We have defined the names of the columns of the sheet and this sheet will be uploaded for data entry.
     * ----------------------------------------------------------------------------------------------------
     * @params - None
     * @return - download the excel sheet with the Indicator Data Entries
     * ----------------------------------------------------------------------------------------------------
     * Author: SUHEL KHAN
     * Date: 11-June-2024
     * PMS TASK ID: 148869
     * ----------------------------------------------------------------------------------------------------
     */
    public function downloadIndicatorsRawData()
    {
        $file_name = date('Y-m-d') . '_indicators_raw_data_template.xlsx';
        $location = 'logs/info-log/' . auth()->user()->email . '/Indicator_data_entries';
        $logData['user'] = auth()->user()->email;
        $logData['file'] = __FILE__;
        $logData['method'] = __FUNCTION__;
        $logData['information'] = "User exported the Indicator Data Entries.";
        CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);

        $user = auth()->user();
        $query = IndicatorDataEntry::query();
        // Check the user role and fetch the districts and departments based on the user role
        if ($user->hasRole('District-User') || $user->hasRole('District-Approver') || $user->hasRole('District-HOD')) {
            // if the user is District-User or District-Approver, then fetch the districts based on the user's district
            $district_id = $user->district_id;
        } elseif ($user->hasRole('Department-User') || $user->hasRole('Department-Approver') || $user->hasRole('Department-HOD')) {
            // if the user is Department-User or Department-Approver, then fetch the departments based on the user's department
            $department_id = $user->department_id;
        }

        // Fetch all the goals
        $goals = CommonHelper::getGoalData('all');

        try {
            $defaultYear = ReportingYear::where('is_default', 'Yes')->first('id');
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            $defaultYear = '';
        }
        /*
        select
            `indicator_data_entries`.`id` as `indicator_data_entries_id`,
            `indicator_data_entries`.`indicator_id`,
            `indicator_data_entries`.`indicator_number`,
            `indicator_data_entries`.`indicator_name`,
            `indicator_data_entries`.`unit`,
            `indicators`.`is_utif`,
            `indicators`.`is_dif`,
            `indicator_data_entries`.`value_type`,
            `indicator_data_entries`.`district_name`,
            `indicator_data_entries`.`department_name`,
            `indicator_data_entries`.`baseline_year`,
            `ide2`.`indicator_value` as `baseline_value`,
            `indicator_data_entries`.`target_value`,
            `indicator_data_entries`.`indicator_value`,
            `indicator_data_entries`.`status`
        from
            `indicator_data_entries`
            inner join `indicators` on `indicator_data_entries`.`indicator_id` = `indicators`.`id`
        left join `indicator_data_entries` as `ide2` on `indicator_data_entries`.`indicator_id` = `ide2`.`indicator_id`
            and `ide2`.`reporting_year_id` = `indicator_data_entries`.`baseline_year_id`
            and (
                (
                `indicators`.`is_utif` = 'yes'
                and `ide2`.`department_id` = `indicator_data_entries`.`department_id`
                )
                or (
                `indicators`.`is_dif` = 'yes'
                and `ide2`.`district_id` = `indicator_data_entries`.`district_id`
                )
            )
            inner join `targets` on `indicator_data_entries`.`target_id` = `targets`.`target_id`
            inner join `goals` on `indicator_data_entries`.`goal_id` = `goals`.`id`
        where
            `indicator_data_entries`.`reporting_year_id` = 7
            and `indicator_data_entries`.`status` = 'Pending'
            and `goals`.`status` = 'Active'
            and `targets`.`status` = 'Active'
            and `indicators`.`is_discontinue` = 'No'
        order by
            `indicator_id` asc
        */
        // Fetch the Indicator Data Entries based on the selected filters
        if (isset($defaultYear) && !empty($defaultYear)) {
            $query->select(
                'indicator_data_entries.id as indicator_data_entries_id',
                'indicator_data_entries.indicator_id',
                'indicator_data_entries.indicator_number',
                'indicator_data_entries.indicator_name',
                'indicator_data_entries.unit',
                'indicators.is_utif',
                'indicators.is_dif',
                'indicators.is_positive_indicator',
                'indicator_data_entries.value_type',
                'indicator_data_entries.district_name',
                'indicator_data_entries.department_name',
                'indicator_data_entries.baseline_year',
                'ide2.indicator_value as baseline_value',
                'indicator_data_entries.target_value',
                'indicator_data_entries.indicator_value',
                'indicator_data_entries.status'
            );

            $query->where('indicator_data_entries.reporting_year_id', $defaultYear->id);

            // $query->whereNot('indicator_data_entries.status', 'Approved');
            $query->whereIn('indicator_data_entries.status', ['Pending', 'Disapproved']);

            if (isset($district_id) && !empty($district_id)) {
                $query->where('indicator_data_entries.district_id', $district_id);
            } else if (isset($department_id) && !empty($department_id)) {
                $query->where('indicator_data_entries.department_id', $department_id);
            }

            $query->join('indicators', 'indicator_data_entries.indicator_id', '=', 'indicators.id');

            $query->leftJoin('indicator_data_entries as ide2', function ($join) {
                $join->on('indicator_data_entries.indicator_id', '=', 'ide2.indicator_id')
                ->whereColumn('ide2.reporting_year_id', 'indicator_data_entries.baseline_year_id');

                // Adding conditional join based on indicators.is_utif and indicators.is_dif
                $join->where(function ($query) {
                    $query->where(function ($q) {
                        $q->where('indicators.is_utif', 'yes')
                        ->whereColumn('ide2.department_id', 'indicator_data_entries.department_id');
                    })->orWhere(function ($q) {
                        $q->where('indicators.is_dif', 'yes')
                        ->whereColumn('ide2.district_id', 'indicator_data_entries.district_id');
                    });
                });
            });

            $query->join('targets', 'indicator_data_entries.target_id', '=', 'targets.target_id');
            $query->join('goals', 'indicator_data_entries.goal_id', '=', 'goals.id');
            $query->where('goals.status', 'Active');
            $query->where('targets.status', 'Active');

            $query->where('indicators.is_discontinue', 'No');
            $query->orderBy('indicator_id', 'asc');
            try {
                // Fetch the Indicator Data Entries based on the selected filters and set the pagination limit from the .env file
                $indicators = $query->get();

                if (!$indicators->isEmpty()) {
                    //Prepare the excel sheet 
                    $filename = 'Indicator_raw_data_template_' . now()->format('Y-m-d_H-i-s') . '.xlsx';
                    Excel::store(new IndicatorDataEntryTemplate($indicators), 'export/Indicator_raw_data_template/' . $filename);

                    // Generate the full path to the saved file
                    $filePath = storage_path('app/export/Indicator_raw_data_template/' . $filename);

                    return response()->download($filePath, $filename);
                } else {
                    $message = "No data found in Indicator Data Entries.";
                    $message .= "Query: " . $query->toSql();
                    $logData['information'] = $message;
                    CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                    $indicators = [];
                    return redirect()->back()->with('error', "No data found in Indicator Data Entries.");
                }
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                $indicators = [];
                return redirect()->back()->with('error', 'Error while exporting Indicator Data Entries. Please try after sometime.');
            }
        } else {
            $indicators = [];
            return redirect()->back()->with('error', "No data found in Indicator Data Entries.");
        }
    }
    /**END:
     * Function- downloadIndicatorsRawData()
     */

    private function setError($worksheet, $row, $status, $message)
    {
        $worksheet->setCellValue('P' . $row, '');
        $worksheet->getStyle('P' . $row)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('E6B8B7'); // Red
        $worksheet->setCellValue('Q' . $row, $status);
        $worksheet->getStyle('Q' . $row)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('E6B8B7'); // Red
        $worksheet->setCellValue('R' . $row, $message);
        $worksheet->getStyle('R' . $row)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('E6B8B7'); // Red
    }

    private function setSuccess($worksheet, $row, $status, $message)
    {
        $worksheet->setCellValue('P' . $row, 'Accepted');
        $worksheet->getStyle('P' . $row)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('00B050'); // Green
        $worksheet->setCellValue('Q' . $row, $status);
        $worksheet->getStyle('Q' . $row)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('00B050'); // Green
        $worksheet->setCellValue('R' . $row, $message);
        $worksheet->getStyle('R' . $row)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('00B050'); // Green
    }

    /**START:
     * Function- importIndicatorsRawData()
     * @description - This function is used to import the Indicator Data Entries from the excel sheet.
     * Here we are fetching all those indicator data entries which are in pending status for year defalut.
     * We have defined the names of the columns of the sheet and this sheet will be uploaded for data entry.
     * ----------------------------------------------------------------------------------------------------
     * @params - None
     * @return - download the excel sheet with the Indicator Data Entries
     * ----------------------------------------------------------------------------------------------------
     * Author: SUHEL KHAN
     * Date: 11-June-2024
     * PMS TASK ID: 148869
     * ----------------------------------------------------------------------------------------------------
     */
    public function importIndicatorsRawData(Request $request)
    {
        //prepared the log file name and location
        $file_name = date('Y-m-d') . '_import_raw_data.txt';
        $location = 'logs/info-log/' . auth()->user()->email . '/import-raw-data';
        $logData['user'] = auth()->user()->email;
        $logData['information'] = "User imported the Indicator Data Entries.";
        $logData['file'] = __FILE__;
        $logData['method'] = __FUNCTION__;
        CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
        $file = $request->file('DataEntryTemplate');

        $logData['information'] = "Uploaded File Name:" . $file->getPathname() . " File Size:" . $file->getSize();
        CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
        if (!$file) {
            return redirect()->back()->with('error', 'No file uploaded');
        }

        $extension = $file->getClientOriginalExtension();
        if (!in_array($extension, ['xls', 'xlsx'])) {
            return redirect()->back()->with('error', 'Invalid file type. Only .xls and .xlsx files are allowed');
        }

        $reader = \PhpOffice\PhpSpreadsheet\IOFactory::createReaderForFile($file->getPathname());
        $spreadsheet = $reader->load($file->getPathname());
        $worksheet = $spreadsheet->getActiveSheet();
        $rowCount = $worksheet->getHighestRow();

        if ($rowCount > 2000) {
            return redirect()->back()->with('error', 'File has too many rows. Maximum allowed is 2000 at one time');
        }

        if ($rowCount == 1) {
            return redirect()->back()->with('error', 'File is empty');
        }

        if ($file->isValid()) {
            // Save the file in the directory storage/app/import/filename_current_date/
            $filename = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
            $extension = $file->getClientOriginalExtension();
            $currentDate = date('Y-m-d_H-i-s');
            $newFileName = $filename . '_' . $currentDate . '.' . $extension;
            $path = Storage::putFileAs('import/uploaded-files', $file, $newFileName);

            // Load the file
            $spreadsheet = IOFactory::load(storage_path('app/' . $path));
            $worksheet = $spreadsheet->getActiveSheet();

            // Check the first column header
            if ($worksheet->getCell('A1')->getValue() !== 'Indicator Data Entries ID') {
                return redirect()->back()->with('error', 'Invalid format of the sheet. The first column in the sheet is not "Indicator Data Entries ID"');
            }

            // Read the Data Value (Current year) and prepare an array of indicator_data_entries_ids
            $indicatorDataEntriesIds = [];
            // Insert two new columns at the beginning
            $getHighestColumn = $worksheet->getHighestColumn();
            if($worksheet->getCell($getHighestColumn . '1')->getValue() !== 'Message') {
                $worksheet->insertNewColumnBefore('R', 2);

                // Set 'Status' and 'Message' headers
                $styleArray = [
                    'font' => [
                        'bold' => true,
                    ],
                    'fill' => [
                        'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                        'startColor' => [
                            'argb' => 'FFFF00',
                        ],
                    ],
                    'borders' => [
                        'outline' => [
                            'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                            'color' => ['argb' => '000000'],
                        ],
                    ],
                ];

                $worksheet->setCellValue('Q1', 'Status');
                $worksheet->getStyle('Q1')->applyFromArray($styleArray);

                $worksheet->setCellValue('R1', 'Message');
                $worksheet->getStyle('R1')->applyFromArray($styleArray);
            }
            

            $highestRow = $worksheet->getHighestRow();
            for ($row = 2; $row <= $highestRow; $row++) { // row 1 is the header
                $indicatorDataEntriesId = $worksheet->getCell('A' . $row)->getValue();
                if (!empty($indicatorDataEntriesId)) {
                    $indicatorDataEntriesIds[] = $indicatorDataEntriesId;
                } else {
                    $this->setError($worksheet, $row, 'Error', 'Indicator Data Entries ID is empty');
                    $error_flag = 1;
                }
            }

            // Fetch the data for those ids
            $query = IndicatorDataEntry::query();
            $query->whereIn('indicator_data_entries.id', $indicatorDataEntriesIds);
            $query->join('indicators', 'indicator_data_entries.indicator_id', '=', 'indicators.id');
            $query->select('indicator_data_entries.id', 'indicator_data_entries.indicator_id', 'indicator_data_entries.indicator_value', 'indicator_data_entries.reporting_year_id', 'indicator_data_entries.district_id', 'indicator_data_entries.department_id', 'indicator_data_entries.status', 'indicator_data_entries.unit', 'indicator_data_entries.update_count');
            try {
                $fetchedData = $query->get();
                if (!empty($fetchedData)) {
                    $fetchedData = $fetchedData->keyBy('id');
                }
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                $fetchedData = [];
            }

            $highestColumn = $worksheet->getHighestColumn();
            $indicatorValueColumn = '';
            $varienceValueColumn = '';
            for ($column = 'A'; $column !== $highestColumn; ++$column) {
                $cellValue = $worksheet->getCell($column . '1')->getValue();
                if ($cellValue === 'Data Value (Current year)') {
                    $indicatorValueColumn = $column;
                }else if ($cellValue === 'Variance Status'){
                    $varienceValueColumn = $column;
                    break;
                }
            }

            if ($indicatorValueColumn === '') {
                return redirect()->back()->with(
                    'error',
                    'The column "Data Value (Current year)" is not found in the sheet.'
                );
            }

            if ($varienceValueColumn === '') {
                return redirect()->back()->with(
                    'error',
                    'The column "Variance Status" is not found in the sheet.'
                );
            }

            $default_year = ReportingYear::where('is_default', 'Yes')->first('id');

            // Update the data and set the status and message
            $error_flag = 0;
            $success_flag = 0;
            $totalRows = $worksheet->getHighestRow();
            $update_count = '';
            $status = '';
            for ($rowIndex = 2; $rowIndex <= $totalRows; $rowIndex++) {
                $indicatorDataEntriesId = $worksheet->getCell('A' . $rowIndex)->getValue(); // Assuming the ID is in column 'A'

                $data = $fetchedData->find($indicatorDataEntriesId);

                if ($data === null) {
                    $this->setError($worksheet, $rowIndex, 'Error', 'ID not available');
                    $error_flag = 1;
                    continue;
                }

                $user = Auth::user();
                if (($user->hasRole('District-User') || $user->hasRole('District-Approver') || $user->hasRole('District-HOD')) && $data->district_id != $user->district_id) {
                    $this->setError($worksheet, $rowIndex, 'Error', 'This Indicator Data Entries ID is not for your district');
                    $error_flag = 1;
                    continue;
                } elseif (($user->hasRole('Department-User') || $user->hasRole('Department-Approver') || $user->hasRole('Department-HOD')) && $data->department_id != $user->department_id) {
                    $this->setError($worksheet, $rowIndex, 'Error', 'This Indicator Data Entries ID is not for your department');
                    $error_flag = 1;
                    continue;
                }

                if ((($user->hasRole('District-User') || $user->hasRole('District-Approver') || $user->hasRole('District-HOD'))) || (($user->hasRole('Department-User') || $user->hasRole('Department-Approver') || $user->hasRole('Department-HOD')))) {
                    if (strtoLower($data->status) == 'approved' || strtoLower($data->status) == 'completed') {
                        if ($data->update_count >= 3) {
                            $this->setError($worksheet, $rowIndex, 'Error', 'This Indicator reached maximum update count. You may update an approved indicator only 3 times.');
                            $error_flag = 1;
                            continue;
                        } else {
                            $update_count  = $data->update_count + 1;
                            $status = 'Sent_for_level_one_approval';
                        }
                    }else{
                        $status = 'Sent_for_level_one_approval';
                    }
                } else {
                    $status = 'Approved';
                }

                if ($data->reporting_year_id != $default_year['id']) {
                    $this->setError($worksheet, $rowIndex, 'Error', 'This Indicator Data Entries ID is not for current year.');
                    $error_flag = 1;
                    continue;
                }

                $indicatorValue = $worksheet->getCell($indicatorValueColumn . $rowIndex)->getValue();

                if ($indicatorValue == '' || $indicatorValue == null) {
                    $this->setError($worksheet, $rowIndex, 'Error', 'Indicator value is empty');
                    $error_flag = 1;
                    continue;
                }

                if (!is_numeric($indicatorValue)) {
                    $this->setError($worksheet, $rowIndex, 'Error', 'Indicator value is not numeric');
                    $error_flag = 1;
                    continue;
                }

                if ($data->unit == 'Percentage' && ($indicatorValue < 0 || $indicatorValue > 100)) {
                    $this->setError($worksheet, $rowIndex, 'Error', 'Data value must be between 0 and 100 for Percentage unit');
                    $error_flag = 1;
                    continue;
                }
                
                $varianceQuery = IndicatorDataEntry::select(
                        'id',
                        'indicator_id',
                        'indicator_number',
                        'indicator_value',
                        \DB::raw("CASE 
                                    WHEN ABS((indicator_value - {$indicatorValue}) / indicator_value) * 100 > 20 THEN 'false'
                                    ELSE 'true'
                                END AS is_within_20_percent_variance")
                    )
                    ->where('indicator_id', $data->indicator_id)
                    ->whereNotNull('indicator_value');

                if (!empty($data->district_id)) {
                    $varianceQuery->where('district_id', $data->district_id);
                } elseif (!empty($data->department_id)) {
                    $varianceQuery->where('department_id', $data->department_id);
                }

                $varianceQuery->where('status', 'Approved')
                    ->where('id', '!=', $data->id)
                    ->orderBy('reporting_year_id', 'desc')
                    ->limit(1);

                try {
                    $varianceResult = $varianceQuery->first();
                } catch (QueryException $e) {
                    $logData['file'] = __FILE__;
                    $logData['method'] = __FUNCTION__;
                    $logData['query'] = $e->getSql();
                    $logData['exception'] = $e->getMessage();
                    CommonHelper::dataActionLogger($logData, 'exception');
                    $varianceResult = null;
                }

                if(isset($varianceResult) && $varianceResult->count() > 0 && $varianceResult->is_within_20_percent_variance == 'false'){
                    if (isset($varienceValueColumn) ) {
                        $varienceValue = $worksheet->getCell($varienceValueColumn . $rowIndex)->getValue();
                        if (strtolower($varienceValue) === 'accept') {
                            // Proceed with the insertion
                        } elseif (strtolower($varienceValue) === '' || strtolower($varienceValue) === 'ignore') {
                            $this->setError($worksheet, $rowIndex, 'Error', 'A variance of ±20% has been observed compared to the prior year. Please set the variance status as "Accept" to save the data.');
                            $error_flag = 1;
                            continue;
                        } else {
                            $this->setError($worksheet, $rowIndex, 'Error', 'Invalid variance status');
                            $error_flag = 1;
                            continue;
                        }
                    }
                }

                try {
                    if ($status == 'Sent_for_level_one_approval' && (strtoLower($data->status) == 'approved' || strtoLower($data->status) == 'completed')) {
                        IndicatorDataEntry::where('id', $data->id)->update([
                            'indicator_value' => $indicatorValue,
                            'status' => $status,
                            'update_count' => $update_count,
                        ]);

                        if($data->district_id != null || $data->district_id != ''){
                            $districtApprovers = commonHelper::getDistrictApprovers($data->district_id);
                            foreach ($districtApprovers as $districtApprover) {
                                Mail::to($districtApprover->email_address)->send(new ApprovalRequestNotificationMail($user,$data,false,$districtApprover));
                            }
                        }else if($data->department_id != null || $data->department_id != ''){
                            $departmentApprovers = commonHelper::getDepartmentApprovers($data->department_id);
                            foreach ($departmentApprovers as $departmentApprover) {
                                Mail::to($departmentApprover->email_address)->send(new ApprovalRequestNotificationMail($user,$data,$departmentApprover, false));
                            }
                        }
                    } else {
                        IndicatorDataEntry::where('id', $data->id)->update([
                            'indicator_value' => $indicatorValue,
                            'status' => $status,
                            'data_added_by' => $user->id,
                        ]);

                        if($data->district_id != null || $data->district_id != ''){
                            $districtApprovers = commonHelper::getDistrictApprovers($data->district_id);
                            foreach ($districtApprovers as $districtApprover) {
                                Mail::to($districtApprover->email_address)->send(new ApprovalRequestNotificationMail($user,$data,false,$districtApprover));
                            }
                        }else if($data->department_id != null || $data->department_id != ''){
                            $departmentApprovers = commonHelper::getDepartmentApprovers($data->department_id);
                            foreach ($departmentApprovers as $departmentApprover) {
                                Mail::to($departmentApprover->email_address)->send(new ApprovalRequestNotificationMail($user,$data,$departmentApprover, false));
                            }
                        }
                    }

                    $this->setSuccess($worksheet, $rowIndex, 'Success', 'Updated successfully');
                    // Set the value of the status column to 'Sent_to_approver'
                    $worksheet->setCellValue('O' . $rowIndex, $status);

                    // Change the font color to green
                    $worksheet->getStyle('O' . $rowIndex)->getFont()->getColor()->setARGB('00B050');
                    $success_flag = 1;
                } catch (QueryException $e) {
                    $this->setError($worksheet, $rowIndex, 'Error', 'Failed to update data');
                    $error_flag = 1;
                    $logData['file'] = __FILE__;
                    $logData['method'] = __FUNCTION__;
                    $logData['query'] = $e->getSql();
                    $logData['exception'] = $e->getMessage();
                    // Log exception to file with file name and method information
                    CommonHelper::dataActionLogger($logData, 'exception');
                }
            }

            // Save the file in the 'import/response-file' folder
            $responseFileName = 'response_' . $newFileName;
            $responseFilePath = 'import/response-files/' . $responseFileName;
            $fullPath = storage_path('app/' . $responseFilePath);
            $writer = IOFactory::createWriter($spreadsheet, 'Xlsx');
            $writer->save($fullPath);

            $logData['information'] = "Response file created at: " . $fullPath;
            CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);

            // Check if the file exists
            if (!file_exists($fullPath)) {
                return redirect()->back()->with('file_not_found', 'Data imported successfully. But response file not created. Please download the file again and check if values updated or not.');
            }
            // Set the download link in the response
            $downloadLink = url('storage/app/' . $responseFilePath);
            if ($error_flag == 1 && $success_flag == 1) {
                return redirect()->back()->with('warning', $downloadLink);
            } elseif ($error_flag == 1 && $success_flag == 0) {
                return redirect()->back()->with('allError', $downloadLink);
            } elseif ($success_flag == 1 && $error_flag == 0) {
                return redirect()->back()->with('success', $downloadLink);
            } else {
                return redirect()->back()->with('error', 'No data found in the sheet.');
            }
        } else {
            return redirect()->back()->with('error', 'File is not valid.');
        }
    }
    /**END:
     * Function- importIndicatorsRawData()
     */
}
