<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Helpers\CommonHelper;
use App\Models\IndicatorDataEntry;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\Log;
use App\Models\DistrictSdgIndex;

class GraphDashboardController extends Controller
{
    function __construct()
    {
        $this->middleware('permission:View Admin Indicator Dashboard', ['only' => ['index']]);
        $this->middleware('auth');
    }

    /**START
     * Function - index
     * To load the data in the admin user dashboard's charts and filters.
     *
     * This method is used to fetch the data for filters and user dashboard graphs.
     * The module is compatible to login as Super Admin, Admin, District-User, District-Approver, Department-User, Department-Approver.
     * if District-User or District-Approver logs in, then the data will be fetched based on the district, as for department users.
     * Here in this method, we also set the current values of all indicators on the bottom of the chart.
     * Firstly all data will load and then ajax request has been sent one by one for all the charts.
     * @param $request (Optional) The request object containing the data for the filters.
     *                            If the request object is empty, then the default data will be loaded.
     * @return array              multiple arrays containing the data for the filters and the charts.
     * 
     * --------------------------------------------------------------------------------------------
     * AUTHOR: SUHEL KHAN
     * DATE: 27-May-2024
     * PM ID: #148099
     * CHANGES: FIRST WRITE-UP
     * --------------------------------------------------------------------------------------------
     */
    public function index(Request $request)
    {
        //Initialising the variables
        $districts = [];
        $departments = [];
        $targets = [];
        $indicators = [];
        $indicator_type = 'dif';
        $default_district = '';
        $default_department = '';

        $data = array();
        //prepared the log file name and location
        $file_name = date('Y-m-d') . '_graphp_dashboard.txt';
        $location = 'logs/info-log/' . auth()->user()->email . '/graph-dashboard';
        $logData['user'] = auth()->user()->email;
        $logData['file'] = __FILE__;
        $logData['method'] = __FUNCTION__;
        $logData['information'] = "User accessed the Graph Dashboard page.";
        //using dataActionLogger function to log the information
        CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);

        $user = auth()->user();
        // Check the user role and fetch the districts and departments based on the user role
        if ($user->hasRole('Super Admin') || $user->hasRole('Admin')) {
            // if the user is Super Admin or Admin, then fetch all the districts and departments
            $districts = CommonHelper::getDistricts('all');
            if (!$districts) {
                $districts = [];
            }

            $departments = CommonHelper::getDepartments('all');
            if (!$departments) {
                $departments = [];
            }

            if ($request->has('district') && !empty($request->input('district'))) {
                $default_district = CommonHelper::getDistricts($request->input('district'));
                if (!$default_district) {
                    $default_district = '';
                }
            } else {
                $default_district = CommonHelper::getDistricts('default');
                if (!$default_district) {
                    $default_district = '';
                }
            }

            $default_department = CommonHelper::getDepartments('default');
            if (!$default_department) {
                $default_department = '';
            }
        } elseif ($user->hasRole('District-User') || $user->hasRole('District-Approver') || $user->hasRole('District-HOD')) {
            // if the user is District-User or District-Approver, then fetch the districts based on the user's district
            $default_district = CommonHelper::getDistricts($user->district_id);
            if (!$default_district) {
                $default_district = '';
            }

            $indicator_type = 'dif';
        } elseif ($user->hasRole('Department-User') || $user->hasRole('Department-Approver') || $user->hasRole('Department-HOD')) {
            // if the user is Department-User or Department-Approver, then fetch the departments based on the user's department
            $default_department = CommonHelper::getDepartments($user->department_id);
            if (!$default_department) {
                $default_department = '';
            }
            $indicator_type = 'utif';
        }

        if ($request->has('type') && !empty($request->input('type'))) {
            $indicator_type = $request->input('type');
        }

        // Fetch the targets based on the selected goal
        if ($request->has('goal') && !empty($request->input('goal'))) {
            $targets = CommonHelper::getTargets('all', $request->input('goal'));
            $targets = collect($targets);
            if ($targets->isEmpty()) {
                $targets = [];
                $default_target = '';
            } else {
                $default_target = $targets[0]->target_id;
            }

            $default_goal = CommonHelper::getGoalData($request->input('goal'));
            if (!$default_goal) {
                $default_goal = '';
            }
        } else {
            // Fetch all the targets
            $targets = CommonHelper::getTargets('all', '1');
            if ($targets) {
                $default_target = $targets[0]->target_id;
            } else {
                $default_target = '';
            }
            $default_goal = CommonHelper::getGoalData('1');
            if (!$default_goal) {
                $default_goal = '';
            }
        }
        // Fetch all the goals
        $goals = CommonHelper::getGoalData('all');
        if (!$goals) {
            $goals = [];
        }

        //Condition prepared to fetch the indicators for indicator filter based on the selected goal, target and type (dif/utif)
        if (($request->has('target') && !empty($request->input('target'))) && ($request->has('goal') && !empty($request->input('goal'))) && ($request->has('type') && !empty($request->input('type')))) {
            $indicators = CommonHelper::getIndicators('all', $request->input('goal'), $request->input('target'), $request->input('type'),'yes');
            if (!$indicators) {
                $indicators = [];
            }
            $default_target = $request->input('target');
        } else if (($request->has('target') && !empty($request->input('target'))) && ($request->has('goal') && !empty($request->input('goal')))) {
            $indicators = CommonHelper::getIndicators('all', $request->input('goal'), $request->input('target'),null,'yes');
            if (!$indicators) {
                $indicators = [];
            }
            $default_target = $request->input('target');
        } else if (($request->has('target') && !empty($request->input('target'))) && ($request->has('type') && !empty($request->input('type')))) {
            $indicators = CommonHelper::getIndicators('all', '', $request->input('target'), $request->input('type'),'yes');
            if (!$indicators) {
                $indicators = [];
            }
            $default_target = $request->input('target');
        } else if (($request->has('target') && !empty($request->input('target')))) {
            $indicators = CommonHelper::getIndicators('all', '', $request->input('target'),null,'yes');
            if (!$indicators) {
                $indicators = [];
            }
            $default_target = $request->input('target');
        } else {
            $indicators = CommonHelper::getIndicators('all', $default_goal['goal_id'], $default_target, $indicator_type,'yes');
            if (!$indicators) {
                $indicators = [];
            }
        }

        // Fetch the indicator ids to pass in the query for fetching the current values of all the fetched indicators.
        $indicator_ids = [];
        if (!empty($indicators) && count($indicators) > 0) {
            foreach ($indicators as $indicator) {
                $indicator_ids[] = $indicator->id;
            }
        } else {
            $indicators = [];
        }

        $indicator_data = [];

        // query to fetch the current indicator values
        $indicator_data_query = IndicatorDataEntry::query();

        $current_year = CommonHelper::getDefaultYear();
        if (!$current_year) {
            $current_year = '';
        } else {
            $current_year = $current_year . '-' . $current_year + 1;
        }

        // Condition prepared to fetch the data based on the selected district or department
        $indicator_data_query->where('reporting_year', $current_year);
        if ($indicator_type == 'dif' && !empty($default_district)) {
            $indicator_data_query->where('district_id', $default_district['id']);
        } elseif ($indicator_type == 'utif' && !empty($default_department)) {
            $indicator_data_query->where('department_id', $default_department['id']);
        }
        // Condition prepared to fetch the data based on the selected indicators
        if (!empty($indicator_ids) && count($indicator_ids) > 0) {
            $indicator_data_query->whereIn('indicator_id', $indicator_ids);
        }

        $indicator_data_query->select('indicator_id', 'indicator_value', 'indicator_normalised_value', 'reporting_year');

        try {
            $indicators_data = $indicator_data_query->get();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');

            $exception = $e->getMessage();
            $logData['information'] = 'Error occurred while fetching the data,Exception: ' . $exception . ' Query: ' . $e->getSql();
            CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
            $indicators_data = [];

            return view('admin_pages.graphDashboard.index', ['districts' => $districts, 'departments' => $departments, 'targets' => $targets, 'indicators' => $indicators, 'goals' => $goals, 'user' => $user, 'indicator_type' => $indicator_type, 'indicator_data' => $indicator_data, 'default_district' => $default_district, 'default_department' => $default_department, 'default_goal' => $default_goal, 'default_target' => $default_target])->with('error', 'Error occurred while fetching the Indicators, Please try again');
        }

        // Fetch the current values of all the fetched indicators
        if (!empty($indicators_data) && $indicators_data->count() > 0) {
            foreach ($indicators_data as $data) {
                // if the indicator type is 'utif' then fetch the indicator value, else fetch the normalised value
                if ($request->has('type') && !empty($request->input('type')) && $request->input('type') == 'utif') {
                    $indicator_data[$data->indicator_id] = $data->indicator_value;
                } else {
                    $indicator_data[$data->indicator_id] = $data->indicator_normalised_value;
                }
            }
        } else {
            $indicator_data = [];
        }

            return view('admin_pages.graphDashboard.index', ['districts' => $districts, 'departments' => $departments, 'targets' => $targets, 'indicators' => $indicators, 'goals' => $goals, 'user' => $user, 'indicator_type' => $indicator_type, 'indicator_data' => $indicator_data, 'default_district' => $default_district, 'default_department' => $default_department, 'default_goal' => $default_goal, 'default_target' => $default_target]);
    }
    /*END - function - Index*/

    /*START
    * Function - getIndicators
    * To fetch the indicators based on the selected goal and target.
    *
    * This method is used to fetch the indicators based on the selected goal and target.
    * @param $request (Optional) The request object containing the data for the filter.
    * @return array              json data for indicators.
    * 
    * --------------------------------------------------------------------------------------------
    * AUTHOR: SUHEL KHAN
    * DATE: 27-May-2024
    * PM ID: #148099
    * CHANGES: FIRST WRITE-UP
    * --------------------------------------------------------------------------------------------
    */
    public function getIndicators(Request $request)
    {
        $indicators = CommonHelper::getIndicators('all', '', $request->input('target'), $request->input('type'),'yes');
        if (!$indicators) {
            $indicators = [];
        }
        return response()->json($indicators);
    }
    /*END - function - getIndicators*/


    /*START
    * Function - getIndicatorsGraphData
    * To fetch the data for the selected indicator to load in the dashboard charts.
    * Here in this method, we fetch the data for the selected indicator to load in the dashboard charts.
    * We also validated the user, if the user is not valid then we return the error message.
    * Here we get two request 1 for DIF SDG for default goal, district or specific goal and District and we return the sdg score for the same.
    * We second another request for data of specific indicator, we also get ajax request for this and we return the data for the same.
    * @param $request  -> IndicatorId, SDG Flag, Type, District, Department, Goal, Target
    * @return array    -> json data for the selected indicator.
    * --------------------------------------------------------------------------------------------
    * AUTHOR: SUHEL KHAN
    * DATE: 28-May-2024
    * PM ID: #148099
    * CHANGES: FIRST WRITE-UP
    * --------------------------------------------------------------------------------------------
    */
    public function getIndicatorsGraphData(Request $request)
    {
        // Initialising the variables
        $response_data_array = [];
        $default_goal_id = '';
        $sdg_index_data = [];
        $indicator_graph_data = [];
        $indicator_data = [];


        // Initialising the variables for log
        $file_name = date('Y-m-d') . '_graphp_dashboard_charts.txt';
        $location = 'logs/info-log/' . auth()->user()->email . '/graph-dashboard';
        $logData['user'] = auth()->user()->email;
        $logData['file'] = __FILE__;
        $logData['method'] = __FUNCTION__;

        // Fetch the indicator data based on the selected indicator
        $indicator_id = $request->query('indicator');
        $sdg_flag = $request->query('sdg');
        // Check if the indicator id is empty and the sdg flag is 0, then return the error message
        if (empty($indicator_id) && $sdg_flag == 0) {
            $response_data_array['status'] = 'error';
            $response_data_array['message'] = 'Indicator ID is required';
            $logData['information'] = 'Indicator ID is required to load the data in the dashboard charts';
            CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
            return response()->json($response_data_array);
        }

        $type = trim($request->query('type'));

        // Prepare the query to fetch the data based on the selected type (DIF/UTIF)
        $query = IndicatorDataEntry::query();
        $user = auth()->user();
        // Check the indicator type and prepare the conditions according to that, if type is DIF then fetch the data based on the district, if type is UTIF then fetch the data based on the department.
        if ($type == 'dif') {
            // Check if the user is District-User or District-Approver, then fetch the data based on the district
            $district_id = trim($request->query('district'));
            if ($user->hasRole('District-User') || $user->hasRole('District-Approver') || $user->hasRole('District-HOD')) {
                if ($user->district_id == $district_id) {
                    $query->where('district_id', $user->district_id);
                } else {
                    $response_data_array['status'] = 'error';
                    $response_data_array['message'] = 'System could not identify you as a valid district user';
                    $logData['information'] = 'System could not identified this user as a valid district user';
                    CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                    return response()->json($response_data_array);
                }
            } else {
                // Check if the district id is empty, then fetch the default district
                if (empty($district_id)) {
                    $district = CommonHelper::getDistricts('default');
                    if (!$district) {
                        $district = [];
                        $logData['information'] = 'Error while fetching district';
                        CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                        $response_data_array['status'] = 'error';
                        $response_data_array['message'] = 'Error while fetching default district';
                        return response()->json($response_data_array);
                    } else {
                        $query->where('district_id', trim($district['id']));
                        $district_id = $district['id'];
                    }
                } else {
                    $query->where('district_id', trim($district_id));
                }
            }
        } else if ($type == 'utif') {
            // Check if the user is Department-User or Department-Approver, then fetch the data based on the department
            $department_id = $request->query('department');
            if ($user->hasRole('Department-User') || $user->hasRole('Department-Approver') || $user->hasRole('Department-HOD')) {
                if ($user->department_id == $department_id) {
                    $query->where('department_id', $user->department_id);
                } else {
                    $response_data_array['status'] = 'error';
                    $response_data_array['message'] = 'System could not identify you as a valid department user';
                    $logData['information'] = 'System could not identified this user as a valid department user';
                    CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                    return response()->json($response_data_array);
                }
            } else {
                // Check if the department id is empty, then fetch the default department
                if (empty($department_id)) {
                    $department = CommonHelper::getDepartments('default');
                    if (!($department)) {
                        $department = [];
                        $logData['information'] = 'Error while fetching department';
                        CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                        $response_data_array['status'] = 'error';
                        $response_data_array['message'] = 'Error while fetching default department';
                        return response()->json($response_data_array);
                    } else {
                        $query->where('department_id', trim($department['id']));
                        $department_id = $department['id'];
                    }
                } else {
                    $query->where('department_id', trim($department_id));
                }
            }
        } else {
            $response_data_array['status'] = 'error';
            $response_data_array['message'] = 'Invalid Indicator type';
            $logData['information'] = 'Invalid Indicator type';
            CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
            return response()->json($response_data_array);
        }

        // This condition has been added to check if sdg_flag is 0 so that SDG data will not be fetched.
        if ($sdg_flag == 0) {

            $query->where('indicator_id', $indicator_id);

            if ($request->has('goal') && !empty($request->query('goal'))) {
                $query = $query->where('goal_id', $request->query('goal'));
            } else {
                $query->where('goal_id', '1');
                $default_goal_id = '1';
            }

            if ($request->has('target') && !empty($request->query('target'))) {
                $query->where('target_id', $request->query('target'));
            }
            $query->select('indicator_id', 'indicator_value', 'indicator_normalised_value', 'reporting_year', 'indicator_number', 'indicator_name', 'target_value');
            $query->Orderby('reporting_year', 'asc');

            try {
                $indicator_data = $query->get();
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');

                $exception = $e->getMessage();
                $query = $query->toSql();
                $response_data_array['status'] = 'error';
                $response_data_array['message'] = 'Error occurred while fetching the data, Please try again';
                $logData['information'] = 'Error occurred while fetching the data,Exception: ' . $exception . ' Query: ' . $query;
                CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                return response()->json($response_data_array);
            }
        }

        /*This condition to prepare the data for DIF SDG goal score,
         if user is district user or admin select the type DIF then we used to show first chart.
         to load the data in this chart we send first ajax request with SDG flag so that only this code will work.
         Data is being prepared for goal score and being sent as json response. */
        $reporting_years = [];
        if ($type == 'dif' && $sdg_flag == 1) {
            // Prepared the query to fetch the SDG data based on the selected district and goal
            $sdg_query = DistrictSdgIndex::query();
            $sdg_query = $sdg_query->where('district_id', $district_id);
            if (!empty($default_goal_id)) {
                $sdg_query = $sdg_query->where('goal_id', $default_goal_id);
            } else {
                $sdg_query = $sdg_query->where('goal_id', $request->query('goal'));
            }
            $sdg_query->join('goals', 'goals.id', '=', 'district_sdg_indexes.goal_id');
            $sdg_query = $sdg_query->select(
                'district_sdg_indexes.id',
                'goal_id',
                'district_id',
                'score',
                'reporting_year',
                'goals.goal_name'
            );
            $query->OrderBy('reporting_year', 'asc');
            try {
                $sdg_data = $sdg_query->get();
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');

                $exception = $e->getMessage();
                $query = $sdg_query->toSql();
                $logData['information'] = 'Error occurred while fetching the data,Exception: ' . $exception . ' Query: ' . $query;
                CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);

                $response_data_array['status'] = 'error';
                $response_data_array['message'] = 'Error occurred while fetching the data, Please try again';

                return response()->json($response_data_array);
            }
            // Prepare the data for the SDG goal score
            if (!empty($sdg_data) && $sdg_data->count() > 0) {
                foreach ($sdg_data as $data) {
                    if (!isset($response_data_array['goal_name']) && empty($response_data_array['goal_name'])) {
                        $response_data_array['goal_name'] = $data->goal_name;
                    }
                    // to prepare the data for the SDG goal score
                    $sdg_index_data[$data->reporting_year] = $data['score'];
                    // to prepare the reporting years for chart
                    $reporting_years[] = $data->reporting_year;
                    if (!isset($response_data_array['sdg_start_year']) && empty($response_data_array['sdg_start_year'])) {
                        $response_data_array['sdg_start_year'] = $data->reporting_year;
                    }
                }
                $sdg_index_data = implode(',', $sdg_index_data);
                $response_data_array['sdg_data'] = $sdg_index_data;
                if (!empty($reporting_years)) {
                    $response_data_array['reporting_years'] = $reporting_years;
                } else {
                    $response_data_array['status'] = 'error';
                    $response_data_array['message'] = 'System could not identify the reporting years for these Indicators';
                    $logData['information'] = 'Data not available for this indicator';
                    CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                    return response()->json($response_data_array);
                }
                $response_data_array['status'] = 'success';
                $logData['information'] = 'Data loaded successfully for the selected district. District ID: ' . $district_id . ' Indicator-type: ' . $type;
                CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                return response()->json($response_data_array);
            } else {
                $response_data_array['status'] = 'error';
                $response_data_array['message'] = 'Data not found for the selected district';
                $logData['information'] = 'Data not available for selected District';
                CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                return response()->json($response_data_array);
            }
        }

        // Here we are preparing the data for indicator for which system get ajax request, Json response is being sent to load the data in the chart.
        $indicator_target_value = [];
        $reporting_years = [];
        if (!empty($indicator_data) && $indicator_data->count() > 0) {
            foreach ($indicator_data as $data) {
                // if the indicator type is 'utif' then fetch the indicator value, else fetch the normalised value
                $indicator_graph_data[$data->reporting_year] = $data['indicator_value'];

                $indicator_target_value[$data->reporting_year] = $data['target_value'];
                $reporting_years[] = $data->reporting_year;

                if (!isset($response_data_array['indicator_name']) && empty($response_data_array['indicator_name'])) {
                    $response_data_array['indicator_name'] = $data->indicator->indicator_number . ': ' . $data->indicator->indicator_name;
                }
                if (!isset($response_data_array['indicator_start_year']) && empty($response_data_array['indicator_start_year'])) {
                    $response_data_array['indicator_start_year'] = $data->reporting_year;
                }
            }
            // Prepare the data for the selected indicator and send the json response
            if (!empty($indicator_graph_data)) {
                $response_data_array['status'] = 'success';
                $indicator_graph_data = implode(',', $indicator_graph_data);
                $response_data_array['indicator_graph_data'] = $indicator_graph_data;
                $indicator_target_value = implode(',', $indicator_target_value);
                $response_data_array['indicator_target_value'] = $indicator_target_value;
                if (!empty($reporting_years)) {
                    $response_data_array['reporting_years'] = $reporting_years;
                } else {
                    $response_data_array['status'] = 'error';
                    $response_data_array['message'] = 'System could not identify the reporting years for these Indicators';
                    $logData['information'] = 'Data not available for this indicator';
                    CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                    return response()->json($response_data_array);
                }
                $logData['information'] = 'Data loaded successfully for the selected indicator. Indicator ID: ' . $indicator_id . ' Type: ' . $type;
                CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                return response()->json($response_data_array);
            } else {
                $response_data_array['status'] = 'error';
                $response_data_array['message'] = 'Data not available for this indicator';
                $logData['information'] = 'Data not available for this indicator';
                CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
                return response()->json($response_data_array);
            }
        } else {
            $response_data_array['status'] = 'error';
            $response_data_array['message'] = 'Data not available for this indicator';
            $logData['information'] = 'Data not available for this indicator';
            CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
            return response()->json($response_data_array);
        }
    }
    /*END - function - getIndicatorsGraphData*/
}
