<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Helpers\CommonHelper;
use App\Models\IndicatorDataEntry;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class DifIndicatorListingController extends Controller
{

    public function __construct()
    {
        $this->middleware('permission:View DIF Indicator Listing', ['only' => ['index']]);
        $this->middleware('auth');
    }


    /**START
     * FUNCTION: index
     * DESCRIPTION: This function is used to fetch the DIF Indicator Listing data and display it on the page.
     * In this function I am preparing query to prepare a pivot of the indicator's data for all the districts.
     * I am fetching the data from the IndicatorDataEntry table and joining it with the IndicatorDataEntry table as previous to get the previous year's data.
     * Indicator values and normalized values are being compared with previous year of the selected(filter) or default year.
     * Conditions are also being prepared according to the applied filters like goal, target, year, district and indicator.
     * Return the view with the fetched data.
     * Parameters: Request $request -> indicator, district, goal, target, year
     * Return: View -> admin_pages/difIndicatorListing/index
     * -------------------------------------------------------------
     * Author: SUHEL KHAN 
     * Date: 22-May-2024
     * PMS TASK ID: 148718
     * CHANGES: FIRST WRITE UP
     * -------------------------------------------------------------
     */
    public function index(Request $request)
    {
        $districts = [];
        $targets = [];
        $indicators = [];

        $logData = array();
        //prepared the log file name and location
        $file_name = date('Y-m-d') . '_dif_indicator_listing.txt';
        $location = 'logs/info-log/' . auth()->user()->email . '/dif-indicator-listing';
        $logData['user'] = auth()->user()->email;
        $logData['file'] = __FILE__;
        $logData['method'] = __FUNCTION__;
        $logData['information'] = "User accessed the DIF Indicator Listing page.";
        //using dataActionLogger function to log the information
        CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);

        //get all goals
        $goals = CommonHelper::getGoalData('all');

        //select distinct reporting_years from IndicatorDataEntry to show in the years filter.
        /*
        * -------------------------------------------------------------------------
            BOC
            Task #156928 The representation of the year should be the financial year i.e. 2021-2022 throughout the application. 
            Added the toArray function after the query as we need the financial year in the form of attay. 
            @author Almaaz Ahmed
            @date 18-12-2024 
        * -------------------------------------------------------------------------    
        */
        try {
            $indicatorYearsData = IndicatorDataEntry::select('reporting_year')->distinct()->get()->toArray();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            $indicatorYearsData = [];
        }

        $indicatorYears = array_column($indicatorYearsData, 'reporting_year');

        /*
        * -------------------------------------------------------------------------
            BOC
            Task #156928 The representation of the year should be the financial year i.e. 2021-2022 throughout the application. 
            removed the below code as it was not needed. It was used to create the single year in the format 2022 but now we are using financial year in the format 2022-2023.
            @author Almaaz Ahmed
            @date 18-12-2024 
        * -------------------------------------------------------------------------    
        */

        /*
        select
            indicator_data_entries.id,
            indicator_data_entries.indicator_id,
            indicator_data_entries.indicator_number,
            indicator_data_entries.reporting_year,
            indicator_data_entries.target_value,
            MAX(
                CASE
                WHEN indicator_data_entries.district_id = 19 THEN indicator_data_entries.indicator_value
                ELSE NULL
                END
            ) AS `Srinagar_value`,
            MAX(
                CASE
                WHEN indicator_data_entries.district_id = 19 THEN indicator_data_entries.indicator_normalised_value
                ELSE NULL
                END
            ) AS `Srinagar_normalized_value`,
            MAX(
                CASE
                WHEN indicator_data_entries.district_id = 19 THEN indicator_data_entries.indicator_value_comment
                ELSE NULL
                END
            ) AS `Srinagar_value_comment`,
            MAX(
                CASE
                WHEN indicator_data_entries.district_id = 19 THEN CASE
                    WHEN previous.indicator_value IS NULL THEN ''
                    WHEN indicator_data_entries.indicator_value > previous.indicator_value THEN 'up'
                    WHEN indicator_data_entries.indicator_value < previous.indicator_value THEN 'down'
                    WHEN indicator_data_entries.indicator_value = previous.indicator_value THEN 'straight'
                    ELSE ''
                END
                ELSE NULL
                END
            ) AS `Srinagar_value_trend`,
            MAX(
                CASE
                WHEN indicator_data_entries.district_id = 19 THEN CASE
                    WHEN previous.indicator_normalised_value IS NULL THEN ''
                    WHEN indicator_data_entries.indicator_normalised_value > previous.indicator_normalised_value THEN 'up'
                    WHEN indicator_data_entries.indicator_normalised_value < previous.indicator_normalised_value THEN 'down'
                    WHEN indicator_data_entries.indicator_normalised_value = previous.indicator_normalised_value THEN 'straight'
                    ELSE ''
                END
                ELSE NULL
                END
            ) AS `Srinagar_normalized_trend`,
            MAX(
                CASE
                WHEN indicator_data_entries.district_id = 1 THEN indicator_data_entries.indicator_value
                ELSE NULL
                END
            ) AS `anantnag_value`,
            MAX(
                CASE
                WHEN indicator_data_entries.district_id = 1 THEN indicator_data_entries.indicator_normalised_value
                ELSE NULL
                END
            ) AS `anantnag_normalized_value`,
            MAX(
                CASE
                WHEN indicator_data_entries.district_id = 1 THEN CASE
                    WHEN previous.indicator_value IS NULL THEN ''
                    WHEN indicator_data_entries.indicator_value > previous.indicator_value THEN 'up'
                    WHEN indicator_data_entries.indicator_value < previous.indicator_value THEN 'down'
                    WHEN indicator_data_entries.indicator_value = previous.indicator_value THEN 'straight'
                    ELSE ''
                END
                ELSE NULL
                END
            ) AS `anantnag_value_trend`,
            MAX(
                CASE
                WHEN indicator_data_entries.district_id = 1 THEN CASE
                    WHEN previous.indicator_normalised_value IS NULL THEN ''
                    WHEN indicator_data_entries.indicator_normalised_value > previous.indicator_normalised_value THEN 'up'
                    WHEN indicator_data_entries.indicator_normalised_value < previous.indicator_normalised_value THEN 'down'
                    WHEN indicator_data_entries.indicator_normalised_value = previous.indicator_normalised_value THEN 'straight'
                    ELSE ''
                END
                ELSE NULL
                END
            ) AS `anantnag_normalized_trend`
            from
            `indicator_data_entries`
            left join `indicator_data_entries` as `previous` on `indicator_data_entries`.`indicator_id` = `previous`.`indicator_id`
            and `previous`.`reporting_year` = '2020-2021'
            and `previous`.`district_id` = indicator_data_entries.district_id
            where
            `indicator_data_entries`.`reporting_year` = '2021-2022'
            and `indicator_data_entries`.`approver_id` is not null
            and `indicator_data_entries`.`district_id` IN (1, 19)
            and `indicator_data_entries`.`unit` != 'absolute'
            and `indicator_data_entries`.`status` in ('approved', 'completed')
        group by
            `indicator_data_entries`.`indicator_id`
        Llimit 
            50 offset 0
        */
        $query = IndicatorDataEntry::query();

        $user = auth()->user();
        if ($user->hasRole('District-User') || $user->hasRole('District-Approver') || $user->hasRole('District-HOD')) {
            // if the user is District-User or District-Approver, then fetch the indicator_data_entries based on the user's district_id
            $query->where('indicator_data_entries.district_id', $user->district_id);

            // fetch the districts based on the user's district
            $districts = CommonHelper::getDistricts($user->district_id);
            if ($districts) {
                $districts = [$districts];
            }else{
                $districts = [];
            }
        } else {
            // fetch all the districts.
            $districts = CommonHelper::getdistricts('all');
            if(!$districts){
                $districts = [];
            }
        }
        // prepare the array of select columns for the query
        $selectColumns = [
            'indicator_data_entries.id',
            'indicator_data_entries.indicator_id',
            'indicator_data_entries.indicator_number',
            'indicator_data_entries.indicator_name',
            'indicator_data_entries.reporting_year',
            'indicator_data_entries.target_value',
            'indicator_data_entries.indicator_value_comment'
        ];

        // we are running the loop to prepare the select columns for each district, as we are preparing the pivot of the indicator's data for all the districts.
        foreach ($districts as $district) {
            $districtId = $district->id;
            $districtName = $district->district_name;
            $selectColumns[] = "MAX(CASE WHEN indicator_data_entries.district_id = {$districtId} THEN indicator_data_entries.indicator_value ELSE NULL END) AS `{$districtName}_value`";
            $selectColumns[] = "MAX(CASE WHEN indicator_data_entries.district_id = {$districtId} THEN indicator_data_entries.indicator_normalised_value ELSE NULL END) AS `{$districtName}_normalized_value`";
            $selectColumns[] = "MAX(CASE WHEN indicator_data_entries.district_id = {$districtId} THEN indicator_data_entries.indicator_value_comment ELSE NULL END) AS `{$districtName}_value_comment`";
            $selectColumns[] = "MAX(CASE WHEN indicator_data_entries.district_id = {$districtId} THEN 
                CASE
                    WHEN previous.indicator_value IS NULL THEN ''
                    WHEN indicator_data_entries.indicator_value > previous.indicator_value THEN 'up'
                    WHEN indicator_data_entries.indicator_value < previous.indicator_value THEN 'down'
                    WHEN indicator_data_entries.indicator_value = previous.indicator_value THEN 'straight'
                    ELSE ''
                END ELSE NULL END) AS `{$districtName}_value_trend`";
            $selectColumns[] = "MAX(CASE WHEN indicator_data_entries.district_id = {$districtId} THEN 
                CASE
                    WHEN previous.indicator_normalised_value IS NULL THEN ''
                    WHEN indicator_data_entries.indicator_normalised_value > previous.indicator_normalised_value THEN 'up'
                    WHEN indicator_data_entries.indicator_normalised_value < previous.indicator_normalised_value THEN 'down'
                    WHEN indicator_data_entries.indicator_normalised_value = previous.indicator_normalised_value THEN 'straight'
                    ELSE ''
                END ELSE NULL END) AS `{$districtName}_normalized_trend`";
            $selectColumns[] = "MAX(CASE WHEN indicator_data_entries.district_id = {$districtId} THEN indicator_data_entries.approver_name ELSE NULL END) AS `{$districtName}_approver_name`";
            $selectColumns[] = "MAX(CASE WHEN indicator_data_entries.district_id = {$districtId} THEN indicator_data_entries.reporter_name ELSE NULL END) AS `{$districtName}_reporter_name`";
        }
        $selectString = implode(', ', $selectColumns);
        $query->select(DB::raw($selectString));

        if ($request->has('district') && !empty($request->input('district'))) {
            $query->where('indicator_data_entries.district_id', $request->input('district'));
        }

        //Condition prepared to fetch the indicators for indicator filter based on the selected goal, target and type (dif or dif)
        if ($request->has('goal') && !empty($request->input('goal'))) {
            //Fetch targets according to the applied filter.
            $targets = CommonHelper::getTargets('all', $request->input('goal'));
            if(!$targets)
            {
                $targets = [];
            }

            //to add the condition of goal id in the query
            $query->where('indicator_data_entries.goal_id', $request->input('goal'));
        } else {
            //fetch the all targets
            $targets = CommonHelper::getTargets('all');
            if(!$targets)
            {
                $targets = [];
            }
        }

        if ($request->has('target') && !empty($request->input('target'))) {
            //to add the condition of target_id in the query.
            $query->where('indicator_data_entries.target_id', $request->input('target'));
        }
        $yearToShowData = '';
        $filter_year = ''; // to show the selected year in the filter
        if ($request->has('year') && !empty($request->input('year'))) {
            $reportingYears = '';
            //to add the condition of reporting year in the query.
            $yearToShowData = $reportingYears = $filter_year = $request->input('year');
            $yearToShowData = explode('-', $yearToShowData)[0];
            // $reportingYears = trim($yearToShowData) . "-" . (trim($yearToShowData) + 1);
            $query->where('indicator_data_entries.reporting_year', $reportingYears);
        } else {
            $reportingYears = [];
            //fetch the if from ReportingYear where is_default = 'Yes'
            $year = CommonHelper::getDefaultYear();
            if(!$year)
            {
                $year = '';
            }

            if (isset($year) && !empty($year)) {
                $yearToShowData = $year;
                $reportingYears[] = $filter_year = $year . "-" . ($year + 1);
                $query->where('indicator_data_entries.reporting_year', $reportingYears);
            }
        }

        // to add the condition of indicator_id in the query.
        if ($request->has('indicator') && !empty($request->input('indicator'))) {
            $query->where('indicator_data_entries.indicator_id', $request->input('indicator'));
        }

        // join the IndicatorDataEntry table with the IndicatorDataEntry table as previous to get the previous year's data.
        $query->leftJoin('indicator_data_entries as previous', function ($join) use ($yearToShowData) {
            $join->on('indicator_data_entries.indicator_id', '=', 'previous.indicator_id')
                ->where('previous.reporting_year', '=', $yearToShowData - 1 . '-' . $yearToShowData)
                ->where('previous.district_id', '=', DB::raw('indicator_data_entries.district_id'));
        });

        // add the conditions to the query
        $query->whereNotNull('indicator_data_entries.approver_id');
        $query->whereNotNull('indicator_data_entries.district_id');
        $query->where('indicator_data_entries.unit', '!=', 'Absolute');
        $query->whereIn('indicator_data_entries.status', ['Approved', 'Completed']);
        $query->orderBy('indicator_data_entries.indicator_id', 'asc');
        $query->groupBy('indicator_data_entries.indicator_id');

        // paginate the query
        $paginate = env('INDICATOR_PAGINATION_LIMIT', '50');
        try {
            $indicators = $query->paginate($paginate);
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            $indicators = [];
        }

        $logData['information'] = "DIF Indicator Listing data fetched successfully.";
        CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);
        return view('admin_pages.difIndicatorListing.index', ['districts' => $districts, 'targets' => $targets, 'indicators' => $indicators, 'indicatoryears' => $indicatorYears, 'reportingYears' => $reportingYears, 'goals' => $goals, 'filter_year' => $filter_year]);
    }
    /**END
     * FUNCTION: index
     */
}
