<?php

namespace App\Helpers;

use App\Models\District;
use App\Models\IndicatorDataEntry;
use App\Models\ReportingYear;
use Illuminate\Database\QueryException;
use App\Helpers\CommonHelper;
use App\Models\Indicator;

class PublicPageHelper
{
    /**START
     * Function - getDistrictSdgIndexes
     * Get the district SDG indexes.
     *
     * This method retrieves the district SDG indexes from the database.
     * If a goal ID is passed, it retrieves data for that goal. Otherwise, it retrieves data for all goals.
     * If a district ID is passed, it retrieves data for that district. Otherwise, it retrieves data for all districts.
     * This method also calculates the trend for each district on the basis of previous year.
     * Trend has been displayed as 'up', 'down', or 'straight' arrow images with score.
     *
     * @param  array $lastThreeYears   An array containing the last three years.
     * @param  array $scoreCategories  An array containing the score categories.
     * @param  string $goalId          (Optional) The ID of the goal to retrieve data for.
     * @param  string $districtId       (Optional) The ID of the district to retrieve data for.
     * @return array                   An array containing the district SDG indexes.
     * 
     * --------------------------------------------------------------------------------------------
     * AUTHOR: SUHEL KHAN
     * DATE: 19-April-2024
     * PM ID: #148098
     * CHANGES: FIRST WRITE-UP
     * --------------------------------------------------------------------------------------------
     */
    public static function getDistrictSdgIndexes($lastThreeYears, $scoreCategories, $goalId = "All", $districtId = 'All')
    {
        /*select 
            `districts`.`id`,
            `districts`.`district_name`,
            `district_sdg_indexes`.`goal_id`,
            `goals`.`goal_name`,
            `district_sdg_indexes`.`district_id`,
            `district_sdg_indexes`.`score`,
            `district_sdg_indexes`.`reporting_year_id`,
            `district_sdg_indexes`.`reporting_year`
        from `districts` 
            left join `district_sdg_indexes` 
                on `district_sdg_indexes`.`district_id` = `districts`.`id`
            left join `goals` 
                on `district_sdg_indexes`.`goal_id` = `goals`.`id`
        where `reporting_year` in ('2020-2021', '2021-2022', '2022-2023') and `districts`.`status` = 'Active'
        order by `district_sdg_indexes`.`reporting_year` asc
        */
        $SQLquery = District::query();
        $SQLquery->Leftjoin('district_sdg_indexes', 'district_sdg_indexes.district_id', '=', 'districts.id')
            ->leftjoin('goals', 'district_sdg_indexes.goal_id', '=', 'goals.id')
            ->whereIn('reporting_year', $lastThreeYears)
            ->where('districts.status', 'Active')
            ->select('districts.id', 'districts.district_name', 'district_sdg_indexes.goal_id', 'goals.goal_name', 'district_sdg_indexes.district_id', 'district_sdg_indexes.score', 'district_sdg_indexes.reporting_year_id', 'district_sdg_indexes.reporting_year')
            ->orderBy('district_sdg_indexes.reporting_year', 'asc');

        if ($districtId != 'All') {
            $SQLquery->where('district_sdg_indexes.district_id', $districtId);
        }
        if ($goalId != 'All') {
            $SQLquery->where('district_sdg_indexes.goal_id', $goalId);
        }
        // Fetch the data from the district_sdg_indexes table
        try {
            $result = $SQLquery->get();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information

            $sql = $SQLquery->toSql();
            $bindings = $SQLquery->getBindings();
            // Replace the placeholders with the actual bindings
            foreach ($bindings as $binding) {
                $value = is_numeric($binding) ? $binding : "'$binding'";
                $sql = preg_replace('/\?/', $value, $sql, 1);
            }

            $logData['query'] = $sql;
            CommonHelper::dataActionLogger($logData, 'exception');
            return false;
        }

        // Add the score_category_name to the data
        $sdg_score_data = [];

        if (isset($result) && !empty($result) && count($result) >= 0) {
            foreach ($result as $row) {
                if (!isset($row->reporting_year) || empty($row->reporting_year)) {
                    continue;
                }
                $year = explode('-', $row->reporting_year);
                $score = $row->score;
                $goal_name = $row->goal_name;
                $goal_id = $row->goal_id;

                // If goal not initialized, initialize it
                if (!isset($sdg_score_data[$row->district_id][$goal_id])) {
                    $sdg_score_data[$row->district_id][$goal_id] = [
                        'district_name' => $row->district_name,
                        'goal_id' => $goal_id,
                        'goal_name' => $goal_name,
                        'year' => [],
                    ];
                }

                // Check score category
                foreach ($scoreCategories as $category) {
                    if ($score >= $category->min_value && $score <= $category->max_value) {
                        // Store score data for each year
                        $sdg_score_data[$row->district_id][$goal_id]['year'][$year[0]] = [
                            'score_category_name' => $category->name,
                            'color_code' => $category->color_code,
                            'score' => $score,
                            'trend' => '',
                        ];
                        break;
                    }
                }
            }
        }
        // Calculate trend for each goal
        foreach ($sdg_score_data as &$data) {
            foreach ($data as &$goal) {
                if (!isset($goal) && count($goal) <= 0 || empty($goal)) {
                    continue;
                }
                $years = array_keys($goal['year']);

                // Sort years in ascending order
                sort($years);

                // Calculate trend
                $prev_score = null;
                foreach ($years as $current_year) {
                    if ($prev_score !== null) {
                        if (isset($goal['year'][$current_year - 1])) {
                            $prev_year_score = $goal['year'][$current_year - 1]['score'];
                            if ($goal['year'][$current_year]['score'] > $prev_year_score) {
                                $goal['year'][$current_year]['trend'] = 'up';
                            } elseif ($goal['year'][$current_year]['score'] < $prev_year_score) {
                                $goal['year'][$current_year]['trend'] = 'down';
                            } else {
                                $goal['year'][$current_year]['trend'] = 'straight';
                            }
                        }
                    }
                    $prev_score = $goal['year'][$current_year]['score'];
                }
            }
        }

        return $sdg_score_data;
    }
    /**END -getDistrictSdgIndexes*/

    /**START
     * Function - getDistrictIndexes
     * Get the district indexes.
     *
     * This method retrieves the district indexes from the database.
     * If a district ID is passed, it retrieves data for that district. Otherwise, it retrieves data for all districts.
     * This method also calculates the trend for each district on the basis of previous year.
     * Trend has been displayed as 'up', 'down', or 'straight' arrow images with score.
     * 
     * @param  array $lastThreeYears   An array containing the last three years.
     * @param  array $scoreCategories  An array containing the score categories.
     * @param  string $districtId       (Optional) The ID of the district to retrieve data for.
     * @return array                   An array containing the district indexes.
     * 
     * --------------------------------------------------------------------------------------------
     * AUTHOR: SUHEL KHAN
     * DATE: 19-April-2024
     * PM ID: #148098
     * CHANGES: FIRST WRITE-UP
     * --------------------------------------------------------------------------------------------
     */
    public static function getDistrictIndexes($lastThreeYears, $scoreCategories, $districtId = "All")
    {
        /*
        select 
            `districts`.`id`, 
            `district_indexes`.`score`, 
            `district_indexes`.`reporting_year_id`, 
            `district_indexes`.`reporting_year` 
        from 
            `districts`
        left join  
            `district_indexes` 
        on 
            `district_indexes`.`district_id` = `districts`.`id` 
        where 
            (`reporting_year` in ('2021-2022', '2022-2023', '2023-2024') OR `district_indexes`.`score`is null) and districts.status = 'Active'
        order by 
            `district_indexes`.`reporting_year` desc
        */
        $districtIndexes = District::query();
        $districtIndexes->Leftjoin('district_indexes', 'district_indexes.district_id', '=', 'districts.id')
            ->where(function ($query) use ($lastThreeYears) {
                $query->whereIn('reporting_year', $lastThreeYears)
                    ->orWhereNull('district_indexes.score');
            })
            ->where('districts.status', 'Active')
            ->select('districts.id', 'district_indexes.score', 'district_indexes.reporting_year_id', 'district_indexes.reporting_year')
            ->orderBy('district_indexes.reporting_year', 'asc');

        if ($districtId != "All") {
            $districtIndexes->where('district_indexes.district_id', $districtId);
        }
        try {
            $result = $districtIndexes->get();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['exception'] = $e->getMessage();

            $sql = $districtIndexes->toSql();
            $bindings = $districtIndexes->getBindings();
            // Replace the placeholders with the actual bindings
            foreach ($bindings as $binding) {
                $value = is_numeric($binding) ? $binding : "'$binding'";
                $sql = preg_replace('/\?/', $value, $sql, 1);
            }

            $logData['query'] = $sql;
            CommonHelper::dataActionLogger($logData, 'exception');
            return false;
        }
        $district_score_data = [];

        if (isset($result) && !empty($result) && count($result) > 0) {
            foreach ($result as $row) {
                if (!isset($row->reporting_year) || empty($row->reporting_year)) {
                    continue;
                }
                $year = explode('-', $row->reporting_year);
                $score = $row->score;
                $district_id = $row->id;

                if (!isset($district_score_data[$district_id])) {
                    $district_score_data[$district_id] = [
                        'district_id' => $district_id,
                        'year' => [],
                    ];
                }

                $district_score_data[$district_id]['year'][$year[0]] = [
                    'score' => $score,
                    'score_category_name' => '',
                    'trend' => '',
                ];

                // Fetch the score categories
                // Check score category
                if (!empty($scoreCategories)) {
                    foreach ($scoreCategories as $category) {
                        if ($score >= $category->min_value && $score <= $category->max_value) {
                            // Store score data for each year
                            $district_score_data[$district_id]['year'][$year[0]] = [
                                'score_category_name' => $category->name,
                                'score' => $score,
                                'trend' => '',
                            ];
                            break;
                        }
                    }
                }
            }
        }
        //calculate trend for each district
        if (isset($district_score_data) && !empty($district_score_data) && count($district_score_data) > 0) {
            foreach ($district_score_data as &$data) {
                if (!isset($data) && count($data) <= 0 || empty($data)) {
                    continue;
                }
                $years = array_keys($data['year']);
                sort($years);
                $prev_score = null;
                foreach ($years as $current_year) {
                    if ($prev_score !== null) {
                        if (isset($data['year'][$current_year - 1])) {
                            $prev_year_score = $data['year'][$current_year - 1]['score'];
                            if ($data['year'][$current_year]['score'] > $prev_year_score) {
                                $data['year'][$current_year]['trend'] = 'up';
                            } elseif ($data['year'][$current_year]['score'] < $prev_year_score) {
                                $data['year'][$current_year]['trend'] = 'down';
                            } else {
                                $data['year'][$current_year]['trend'] = 'straight';
                            }
                        }
                    }
                    $prev_score = $data['year'][$current_year]['score'];
                }
            }
        }
        return $district_score_data;
    }
    /**END -getDistrictIndexes*/

    /**START
     * Function - getDistrictCodeWithScoreCategories
     * Get the district code with score categories.
     *
     * This method retrieves the district code with score categories from the database.
     * It also calculates the score category for each district based on the default year.
     * This will set the data in the Map according to the score categories.
     * 
     * @param  array $district_array An array containing the district data.
     * @param  array $sdgData        An array containing the SDG data.
     * @param  string $defaultYear   The default year to generate the last three years from.
     * @return array                 An array containing the district code with score categories.
     * 
     * --------------------------------------------------------------------------------------------
     * AUTHOR: SUHEL KHAN
     * DATE: 19-April-2024
     * PM ID: #148098
     * CHANGES: FIRST WRITE-UP
     * --------------------------------------------------------------------------------------------
     */
    public static function getDistrictCodeWithScoreCategories($district_array, $sdgData, $defaultYear)
    {
        $score_category_district = [];
        $score_category_district['Achiever'] = [];
        $score_category_district['Performer'] = [];
        $score_category_district['Front_Runner'] = [];
        $score_category_district['Aspirant'] = [];
        // Added these districts at no data because these districts(Muzaffarabad and Mirpur) are not available for SDG data but we have to show these districts on the map.
        $score_category_district['nodata'][0] = '991';
        $score_category_district['nodata'][1] = '992';
        foreach ($district_array as $key => $district) {
            if (isset($sdgData[$district['id']]) && !empty($sdgData[$district['id']])) {
                if (isset($sdgData[$district['id']]['year'][$defaultYear]) && !empty($sdgData[$district['id']]['year'][$defaultYear])) {
                    if (strtolower($sdgData[$district['id']]['year'][$defaultYear]['score_category_name']) == 'front_runner') {
                        $score_category_district['Front_Runner'][] = $district['district_code'];
                    } elseif (strtolower($sdgData[$district['id']]['year'][$defaultYear]['score_category_name']) == 'performer') {
                        $score_category_district['Performer'][] = $district['district_code'];
                    } elseif (strtolower($sdgData[$district['id']]['year'][$defaultYear]['score_category_name']) == 'achiever') {
                        $score_category_district['Achiever'][] = $district['district_code'];
                    } else {
                        $score_category_district['Aspirant'][] = $district['district_code'];
                    }
                } else {
                    $score_category_district['nodata'][] = $district['district_code'];
                }
            } else {
                $score_category_district['nodata'][] = $district->district_code;
            }
        }
        return $score_category_district;
    }
    /**END -getDistrictCodeWithScoreCategories*/

    /**START
     * Function - getDistrictIndicators
     * Get the district indicators.
     *
     * This method retrieves the district indicators from the database.
     * If a district ID is passed, it retrieves data for that district. Otherwise, it retrieves data for all districts.
     * If a goal ID is passed, it retrieves data for that goal. Otherwise, it retrieves data for all goals.
     * If a year is passed, it retrieves data for that year. Otherwise, it retrieves data for the default year.
     * This method also calculates the score for each indicator and categorizes it as 'best_performance', 'require_attention', or 'require_immidiate_action'.
     * 
     * @param  string $districtId (Optional) The ID of the district to retrieve data for.
     * @param  string $year       (Optional) The year to retrieve data for.
     * @param  string $goal_id    (Optional) The ID of the goal to retrieve data for.
     * @return array              An array containing the district indicators.
     * 
     * --------------------------------------------------------------------------------------------
     * AUTHOR: SUHEL KHAN
     * DATE: 19-April-2024
     * PM ID: #148098
     * CHANGES: FIRST WRITE-UP
     * --------------------------------------------------------------------------------------------
     */
    public static function getDistrictIndicators($districtId = 'All', $year = 'default', $goal_id = 'All')
    {
        $default_year = '';
        $lastThreeYear = '';
        if ($year == 'default') {
            try {
                $defaultYearData = ReportingYear::where('is_default', 'Yes')->first('reporting_year');
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                return false;
            }
            $defaultYearData = explode('-', $defaultYearData->reporting_year);
            $default_year = $defaultYearData[0];
        } elseif ($year == 'lastThreeYears') {
            try {
                $defaultYearData = ReportingYear::where('is_default', 'Yes')->first('reporting_year');
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                return false;
            }

            $defaultYearData = explode('-', $defaultYearData->reporting_year);
            $defaultYear = $defaultYearData[0];
            $lastThreeYear = CommonHelper::getLastThreeYears($defaultYear);
        } else {
            $default_year = $year;
        }

        /*
        select
            `indicator_data_entries`.`indicator_id`,
            `indicator_data_entries`.`indicator_number`,
            `indicator_data_entries`.`district_name`,
            `indicator_data_entries`.`district_id`,
            `indicator_data_entries`.`reporting_year`,
            `indicator_data_entries`.`unit`,
            `indicator_data_entries`.`target_value`,
            `indicator_data_entries`.`indicator_normalised_value`,
            `indicator_data_entries`.`indicator_value_comment`,
        from
            `indicator_data_entries`
        where
            `indicator_data_entries`.`district_id` = 1
            and `indicator_data_entries`.`reporting_year` = '2020-2021'
            and `indicator_data_entries`.`approver_id` is not null
        */
        $query  = IndicatorDataEntry::query();
        $query->select('indicator_id', 'indicator_number', 'district_name', 'district_id', 'reporting_year', 'unit', 'target_value', 'indicator_normalised_value','indicator_value_comment', 'indicator_name', 'indicator_value');
        $query->whereNotNull('indicator_data_entries.approver_id');
        $query->whereNot('unit', 'Absolute');

        if ($districtId != 'All') {
            $query->where('district_id', $districtId);
        }

        if ($goal_id != 'All') {
            $query->where('goal_id', $goal_id);
        }

        if ($year == 'lastThreeYears') {
            $query->whereIn('reporting_year', $lastThreeYear);
        } else {
            $default_year = $default_year . '-' . ($default_year + 1);
            $query->where('reporting_year', $default_year);
        }
        try {
            $result = $query->get();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['exception'] = $e->getMessage();

            $sql = $query->toSql();
            $bindings = $query->getBindings();
            // Replace the placeholders with the actual bindings
            foreach ($bindings as $binding) {
                $value = is_numeric($binding) ? $binding : "'$binding'";
                $sql = preg_replace('/\?/', $value, $sql, 1);
            }

            $logData['query'] = $sql;
            CommonHelper::dataActionLogger($logData, 'exception');
            return false;
        }

        $district_indicators_data = [];
        foreach ($result as $row) {
            $indicator_score = $row->indicator_normalised_value;
            if ($indicator_score >= env('BEST_PERMORMANCE_MIN_VALUE', 74) && $indicator_score <= env('BEST_PERMORMANCE_MAX_VALUE', 100)) {
                $district_indicators_data['best_performance'][$row->indicator_id]['indicator_number'] = trim($row->indicator_number);
                $district_indicators_data['best_performance'][$row->indicator_id]['indicator_name'] = trim($row->indicator_name);
                $district_indicators_data['best_performance'][$row->indicator_id]['indicator_value'] = trim($row->indicator_value);
            } elseif ($indicator_score >= env('REQUIRE_ATTENTION_MIN_VALUE', 64) && $indicator_score <= env('REQUIRE_ATTENTION_MAX_VALUE', 73)) {
                $district_indicators_data['require_attention'][$row->indicator_id]['indicator_number'] = trim($row->indicator_number);
                $district_indicators_data['require_attention'][$row->indicator_id]['indicator_name'] = trim($row->indicator_name);
                $district_indicators_data['require_attention'][$row->indicator_id]['indicator_value'] = trim($row->indicator_value);
            } else {
                $district_indicators_data['require_immidiate_action'][$row->indicator_id]['indicator_number'] = trim($row->indicator_number);
                $district_indicators_data['require_immidiate_action'][$row->indicator_id]['indicator_name'] = trim($row->indicator_name);
                $district_indicators_data['require_immidiate_action'][$row->indicator_id]['indicator_value'] = trim($row->indicator_value);
            }
        }
        return $district_indicators_data;
    }
    /**END -getDistrictIndicators*/

    /**START
     * Function - getGoalDifIndicatorData
     * Get the goal DIF indicator data.
     *
     * This method retrieves the goal DIF indicator data from the database.
     * If a goal ID is passed, it retrieves data for that goal. Otherwise, it retrieves data for all goals.
     * If a year is passed, it retrieves data for that year. Otherwise, it retrieves data for the default year.
     * 
     * @param  string $goal_id   (Optional) The ID of the goal to retrieve data for.
     * @param  string $year      (Optional) The year to retrieve data for.
     * @param  array $districts  An array containing the district data.
     * @return array             An array containing the goal DIF indicator data.
     * 
     * --------------------------------------------------------------------------------------------
     * AUTHOR: SUHEL KHAN
     * DATE: 19-April-2024
     * PM ID: #148098
     * CHANGES: FIRST WRITE-UP
     * --------------------------------------------------------------------------------------------
     */
    public static function getGoalDifIndicatorData($goal_id, $year)
    {
        /*
        select
            `indicator_data_entries`.`district_id`,
            `indicator_data_entries`.`indicator_value`,
            `indicator_data_entries`.`indicator_id`,
            `indicator_data_entries`.`indicator_number`,
            `indicator_data_entries`.`indicator_name`,
            `indicator_data_entries`.`reporting_year`
        from
            `indicator_data_entries`
        where
            `indicator_data_entries`.`goal_id` = 1
            and `indicator_data_entries`.`reporting_year` ='2023-2024'
            and indicator_data_entries.district_id is not null 
            and indicator_data_entries.approver_id is not null
        order by
            `indicator_data_entries`.`indicator_id` desc
        */
        $year = $year . '-' . $year + 1;
        $query = IndicatorDataEntry::query();

        $query->where('indicator_data_entries.goal_id', $goal_id)
            ->where('indicator_data_entries.reporting_year', $year)
            ->whereNotNull('indicator_data_entries.district_id')
            ->whereNotNull('indicator_data_entries.approver_id')
            ->select('indicator_data_entries.district_id', 'indicator_data_entries.indicator_value','indicator_data_entries.indicator_value_comment', 'indicator_data_entries.indicator_id', 'indicator_data_entries.indicator_number', 'indicator_data_entries.indicator_name', 'indicator_data_entries.reporting_year', 'indicator_data_entries.unit')
            ->orderBy('indicator_data_entries.indicator_id', 'asc');
        try {
            $result = $query->get();
            // dd($query->toSql(), $query->getBindings());
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['exception'] = $e->getMessage();

            $sql = $query->toSql();
            $bindings = $query->getBindings();
            // Replace the placeholders with the actual bindings
            foreach ($bindings as $binding) {
                $value = is_numeric($binding) ? $binding : "'$binding'";
                $sql = preg_replace('/\?/', $value, $sql, 1);
            }

            $logData['query'] = $sql;
            CommonHelper::dataActionLogger($logData, 'exception');
            return false;
        }
        $goal_dif_indicator_data = [];
        if (isset($result) && !empty($result)) {
            $goal_dif_indicator_data = [];
            foreach ($result as $row) {
                if (!empty($row->district_id)) {
                    $goal_dif_indicator_data[$row->indicator_id]['indicator_number'] = trim($row->indicator_number);
                    $goal_dif_indicator_data[$row->indicator_id]['indicator_name'] = trim($row->indicator_name);
                    $goal_dif_indicator_data[$row->indicator_id]['unit'] = trim($row->unit);
                    $goal_dif_indicator_data[$row->indicator_id][$row->district_id]['value'] = trim($row->indicator_value);
                    $goal_dif_indicator_data[$row->indicator_id][$row->district_id]['comment'] = trim($row->indicator_value_comment);
                }
            }
        }
        return $goal_dif_indicator_data;
    }
    /**END -getGoalDifIndicatorData*/

    /**START
     * Function - getGoalUtifIndicatorData
     * Get the goal UTIF indicator data.
     *
     * This method retrieves the goal UTIF indicator data from the database.
     * If a goal ID is passed, it retrieves data for that goal. Otherwise, it retrieves data for all goals.
     * If a year is passed, it retrieves data for that year. Otherwise, it retrieves data for the default year.
     * Trend is also being calculated in this fiunction to display on frontend.
     * 
     * @param  string $goal_id   (Optional) The ID of the goal to retrieve data for.
     * @param  string $year      (Optional) The year to retrieve data for.
     * @param  array $districts  An array containing the district data.
     * @return array             An array containing the goal UTIF indicator data.
     * 
     * --------------------------------------------------------------------------------------------
     * AUTHOR: SUHEL KHAN
     * DATE: 19-April-2024
     * PM ID: #148098
     * CHANGES: FIRST WRITE-UP
     * --------------------------------------------------------------------------------------------
     */
    public static function getGoalIndicatorDataforLastThreeYears($goal_id, $lastThreeYear, $type, $district_id = '')
    {
        /*
        select 
            `indicator_data_entries`.`indicator_value`,
            `indicator_data_entries`.`indicator_id`, 
            `indicator_data_entries`.`indicator_number`,
            `indicator_data_entries`.`indicator_name`, 
            `indicator_data_entries`.`reporting_year`
        from 
            `indicator_data_entries` 
        where 
            `indicator_data_entries`.`goal_id` = 10 
            and `indicator_data_entries`.`reporting_year` in ('2020-2021', '2021-2022', '2022-2023') 
            and `indicator_data_entries`.`approver_id` is not null 
            and `indicator_data_entries`.`department_id` is not null 
        order by 
            `indicator_data_entries`.`reporting_year` asc
        */
        $query = IndicatorDataEntry::query();

        $query->where('indicator_data_entries.goal_id', $goal_id)
            ->whereIn('indicator_data_entries.reporting_year', $lastThreeYear)
            ->whereNotNull('indicator_data_entries.approver_id')
            ->join('indicators', 'indicators.id', '=', 'indicator_data_entries.indicator_id')
            ->select('indicator_data_entries.indicator_value', 'indicator_data_entries.indicator_id','indicator_data_entries.indicator_value_comment', 'indicator_data_entries.indicator_number', 'indicator_data_entries.indicator_name', 'indicator_data_entries.reporting_year', 'indicator_data_entries.baseline_year', 'indicator_data_entries.unit', 'indicators.is_positive_indicator')
            ->orderBy('indicator_data_entries.indicator_id')
            ->orderBy('indicator_data_entries.reporting_year', 'asc');
        if ($type == 'utif') {
            $query->whereNotNull('indicator_data_entries.department_id');
        } else {
            $query->whereNotNull('indicator_data_entries.district_id');
            if ($district_id != '') {
                $query->where('indicator_data_entries.district_id', $district_id);
            }
        }
        try {
            $result = $query->get();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['exception'] = $e->getMessage();

            $sql = $query->toSql();
            $bindings = $query->getBindings();
            // Replace the placeholders with the actual bindings
            foreach ($bindings as $binding) {
                $value = is_numeric($binding) ? $binding : "'$binding'";
                $sql = preg_replace('/\?/', $value, $sql, 1);
            }

            $logData['query'] = $sql;
            CommonHelper::dataActionLogger($logData, 'exception');
            return false;
        }

        $goal_utif_indicator_data = [];
        if (isset($result) && !empty($result)) {
            $goal_utif_indicator_data = [];
            foreach ($result as $row) {
                $year = explode('-', $row->reporting_year); // Extract the year from the reporting_year

                $goal_utif_indicator_data[$row->indicator_id]['indicator_number'] = trim($row->indicator_number);
                $goal_utif_indicator_data[$row->indicator_id]['indicator_name'] = trim($row->indicator_name);
                $goal_utif_indicator_data[$row->indicator_id]['unit'] = trim($row->unit);
                $goal_utif_indicator_data[$row->indicator_id]['baseline_year'] = $row->baseline_year;
                $goal_utif_indicator_data[$row->indicator_id][$year[0]]['score'] = $row->indicator_value;
                $goal_utif_indicator_data[$row->indicator_id][$year[0]]['comment'] = $row->indicator_value_comment;

                // Calculate the trend
                $previousYear = $year[0] - 1;
                if (isset($goal_utif_indicator_data[$row->indicator_id][$previousYear]['score']) && !empty($goal_utif_indicator_data[$row->indicator_id][$previousYear]['score'])) {
                    if (!empty($row->indicator_value)  && $row->indicator_value > $goal_utif_indicator_data[$row->indicator_id][$previousYear]['score']) {
                        if(strtolower($row->is_positive_indicator) == 'negative')
                        {
                            $goal_utif_indicator_data[$row->indicator_id][$year[0]]['trend'] = 'down';
                        }else{
                            $goal_utif_indicator_data[$row->indicator_id][$year[0]]['trend'] = 'up';
                        }
                    } elseif (!empty($row->indicator_value)  && $row->indicator_value < $goal_utif_indicator_data[$row->indicator_id][$previousYear]['score']) {
                        if(strtolower($row->is_positive_indicator) == 'negative')
                        {
                            $goal_utif_indicator_data[$row->indicator_id][$year[0]]['trend'] = 'up';
                        }else{
                            $goal_utif_indicator_data[$row->indicator_id][$year[0]]['trend'] = 'down';
                        }
                    } elseif (!empty($row->indicator_value) && $row->indicator_value == $goal_utif_indicator_data[$row->indicator_id][$previousYear]['score']) {
                        $goal_utif_indicator_data[$row->indicator_id][$year[0]]['trend'] = 'straight';
                    } else {
                        $goal_utif_indicator_data[$row->indicator_id][$year[0]]['trend'] = '';
                    }
                }
            }
        }
        /*select 
            distinct `indicator_id`, 
            `indicator_value` 
        from 
            `indicators` 
        inner join 
            `indicator_data_entries` 
            on `indicators`.`id` = `indicator_data_entries`.`indicator_id` 
            and `indicators`.`baseline_year_id` = `indicator_data_entries`.`reporting_year_id` 
        where 
            `indicator_data_entries`.`goal_id` = 1 
            and `indicator_data_entries`.`district_id` is not null 
            and `indicator_data_entries`.`district_id` = 19
        */
        $baseLineDataQuery = Indicator::query();
        $baseLineDataQuery->select('indicator_id', 'indicator_value');
        $baseLineDataQuery->join('indicator_data_entries', function ($join) {
            $join->on('indicators.id', '=', 'indicator_data_entries.indicator_id')
                ->on('indicators.baseline_year_id', '=', 'indicator_data_entries.reporting_year_id');
        });
        $baseLineDataQuery->where('indicator_data_entries.goal_id', $goal_id);
        if ($type == 'utif') {
            $baseLineDataQuery->whereNotNull('indicator_data_entries.department_id');
        } else {
            $baseLineDataQuery->whereNotNull('indicator_data_entries.district_id');
            if ($district_id != '') {
                $baseLineDataQuery->where('indicator_data_entries.district_id', $district_id);
            }
        }
        $baseLineDataQuery->distinct();
        $baselineValue = $baseLineDataQuery->get()->keyBy('indicator_id');
        if (isset($baselineValue) && !$baselineValue->isEmpty()) {
            foreach ($goal_utif_indicator_data as $indicator_id => &$indicator) {
                if ($baselineValue->has($indicator_id)) {
                    $indicator['baseline_value'] = $baselineValue->get($indicator_id)->indicator_value;
                } else {
                    $indicator['baseline_value'] = '';
                }
            }
        } else {
            foreach ($goal_utif_indicator_data as $indicator_id => &$indicator) {
                $indicator['baseline_value'] = '';
            }
        }
        return $goal_utif_indicator_data;
    }
    /**END -getGoalUtifIndicatorData*/


    /**START
     * Function - getSunriseGraphData
     * Get the sunrise graph data.
     *
     * This method retrieves the sunrise graph data from the database.
     * If a district ID is passed, it retrieves data for that district. Otherwise, it retrieves data for all districts.
     * If a year is passed, it retrieves data for that year. Otherwise, it retrieves data for the default year.
     * This method also calculates the score for each indicator and categorizes it as 'best_performance', 'require_attention', or 'require_immidiate_action'.
     * 
     * @param  string $districtId (Optional) The ID of the district to retrieve data for.
     * @param  string $year       (Optional) The year to retrieve data for.
     * @param  array $goals       An array containing the goal data.
     * @return array              An array containing the sunrise graph data.
     * 
     * --------------------------------------------------------------------------------------------
     * AUTHOR: SUHEL KHAN
     * DATE: 15-May-2024
     * PM ID: #149447
     * CHANGES: FIRST WRITE-UP
     * --------------------------------------------------------------------------------------------
     */
    public static function getSunriseGraphData($districtId = 'All', $year = 'default', $goals = array())
    {
        $default_year = '';
        $lastThreeYear = '';
        // Get the default year
        if ($year == 'default') {
            try {
                $defaultYearData = ReportingYear::where('is_default', 'Yes')->first('reporting_year');
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                return false;
            }
            $defaultYearData = explode('-', $defaultYearData->reporting_year);
            $default_year = $defaultYearData[0];
        } elseif ($year == 'lastThreeYears') {
            // if year is lastThreeYears then get the last three years
            try {
                $defaultYearData = ReportingYear::where('is_default', 'Yes')->first('reporting_year');
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                return false;
            }

            $defaultYearData = explode('-', $defaultYearData->reporting_year);
            $defaultYear = $defaultYearData[0];
            $lastThreeYear = CommonHelper::getLastThreeYears($defaultYear);
        } else {
            $default_year = $year;
        }

        /*
        select
            `indicator_data_entries`.`indicator_id`,
            `indicator_data_entries`.`indicator_number`,
            `indicator_data_entries`.`district_name`,
            `indicator_data_entries`.`district_id`,
            `indicator_data_entries`.`reporting_year`,
            `indicator_data_entries`.`unit`,
            `indicator_data_entries`.`target_value`,
            `indicator_data_entries`.`indicator_normalised_value`,
        from
            `indicator_data_entries`
        where
            `indicator_data_entries`.`district_id` = 1
            and `indicator_data_entries`.`reporting_year` = '2020-2021'
            and `indicator_data_entries`.`status` = 'Approved'
        */
        // Get the data from the indicator_data_entries table for the sunrise graph
        $query  = IndicatorDataEntry::query();
        $query->select('indicator_id', 'indicator_number', 'district_name', 'district_id', 'reporting_year', 'unit', 'target_value', 'indicator_normalised_value', 'indicator_name', 'goal_id');
        $query->where('indicator_data_entries.status','Approved');
        $query->WhereNot('unit', 'Absolute');

        // Check if district ID is passed
        if ($districtId != 'All') {
            $query->where('district_id', $districtId);
        }

        // Check if year is lastThreeYears else get the default year
        if ($year == 'lastThreeYears') {
            $query->whereIn('reporting_year', $lastThreeYear);
        } else {
            $default_year = $default_year . '-' . ($default_year + 1);
            $query->where('reporting_year', $default_year);
        }
        try {
            $result = $query->get();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['exception'] = $e->getMessage();

            $sql = $query->toSql();
            $bindings = $query->getBindings();
            // Replace the placeholders with the actual bindings
            foreach ($bindings as $binding) {
                $value = is_numeric($binding) ? $binding : "'$binding'";
                $sql = preg_replace('/\?/', $value, $sql, 1);
            }

            $logData['query'] = $sql;
            CommonHelper::dataActionLogger($logData, 'exception');
            return false;
        }

        // prepapre the data for the sunrise graph based on the score and the score categories
        if (isset($result) && !empty($result) && count($result) > 0) {
            $sunrise_graph_data = [];
            foreach ($goals as $goal) {
                $bestperformance_count = 0;
                $immidiate_action_count = 0;
                $require_attention_count = 0;
                foreach ($result as $row) {
                    if ($goal['goal_id'] == $row->goal_id) {
                        $indicator_score = $row->indicator_normalised_value;
                        // check if the score is in the best performance, require attention or require immidiate action
                        if ($indicator_score >= env('BEST_PERMORMANCE_MIN_VALUE', 74) && $indicator_score <= env('BEST_PERMORMANCE_MAX_VALUE', 100)) {
                            $sunrise_graph_data['best_performance'][$goal['goal_id']] = ++$bestperformance_count;
                        } elseif ($indicator_score >= env('REQUIRE_ATTENTION_MIN_VALUE', 64) && $indicator_score <= env('REQUIRE_ATTENTION_MAX_VALUE', 73)) {
                            $sunrise_graph_data['require_attention'][$goal['goal_id']] = ++$require_attention_count;
                        } else {
                            $sunrise_graph_data['require_immidiate_action'][$goal['goal_id']] = ++$immidiate_action_count;
                        }
                    }
                }
            }
        } else {
            $sunrise_graph_data = [];
        }
        return $sunrise_graph_data;
    }
    /**END -getSunriseGraphData*/

    /**START
     * Function - determineColor
     * To get the color of the bar for according to score.
     * 
     * --------------------------------------------------------------------------------------------
     * AUTHOR: Almaaz Ahmed
     * DATE: 20-December-2024
     * PM ID: #156546
     * CHANGES: FIRST WRITE-UP
     * --------------------------------------------------------------------------------------------
     */
    public static function determineColor($score)
    {
        if ($score == 100) {
            return '#00aeef';
        } elseif ($score >= 65 && $score <= 99) {
            return '#00a084';
        } elseif ($score >= 50 && $score <= 64) {
            return '#ffc40c';
        } elseif ($score >= 0 && $score <= 49) {
            return '#dd1e47';
        } else {
            return '#dfdfdf';
        }
    }
    /**END -determineColor*/


    /**START
     * Function - getReportingYearsForDashboardInput
     * To get the years before the default year.
     * 
     * --------------------------------------------------------------------------------------------
     * AUTHOR: Almaaz Ahmed
     * DATE: 20-December-2024
     * PM ID: #156546
     * CHANGES: FIRST WRITE-UP
     * --------------------------------------------------------------------------------------------
     */
    public static function getReportingYearsForDashboardInput()
    {
        $defaultYearRecord = app('defaultYear');
        $defaultYear = $defaultYearRecord->reporting_year_base;
        $reportingYears = [];
        try {
            $reportingYears = ReportingYear::where('reporting_year_base', '<=', $defaultYear)
                                       ->orderBy('reporting_year_base', 'asc')
                                       ->get();
        } catch (QueryException $e) {
            $exception_data = [];
            $exception_data['file'] = __FILE__;
            $exception_data['method'] = __METHOD__;
            $exception_data['query'] = $e->getSql();
            $exception_data['exception'] = $e->getMessage();
            CommonHelper::dataActionLogger($exception_data, 'exception');
            return $reportingYears;
        }
        return $reportingYears;
    }
}
