<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\AfterSheet;
use Illuminate\Support\Collection;

class DistrictWiseSdgIndicatorExport implements FromCollection, WithHeadings, WithTitle, WithMapping, WithEvents
{
    protected $data;
    protected $targetCounts;
    protected $indicatorCounts;

    public function __construct(Collection $data, Collection $targetCounts, Collection $indicatorCounts)
    {
        $this->data = $data;
        $this->targetCounts = $targetCounts;
        $this->indicatorCounts = $indicatorCounts;
    }

    public function collection()
    {
        return $this->data;
    }

    public function headings(): array
    {
        return [
            'Sr No.',
            'Goal',
            'Target',
            'Indicator',
            'Unit',
            'Baseline Year',
            'Baseline Value',
            'Year',
            'Taregt 2030',
            'District',
            'Achieved Value (Current Year)',
        ];
    }

    public function map($row): array
    {
        static $srNo = 1;

        return [
            $srNo++,
            $row->goal_name,
            $row->target_name,
            $row->indicator_number . ': ' . $row->indicator_name,
            $row->unit,
            $row->baseline_year,
            $row->baseline_value,
            $row->reporting_year,
            $row->target_value,
            $row->district_name,
            $row->achieved_value,
        ];
    }

    public function title(): string
    {
        return 'SDG Indicators';
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function (AfterSheet $event) {
                $sheet = $event->sheet->getDelegate();
                $highestColumn = $sheet->getHighestColumn();
                $highestRow = $sheet->getHighestRow();

                // Apply array of styles to 'A1' range
                $sheet->getStyle('A1:' . $highestColumn . '1')->applyFromArray([
                    'font' => [
                        'bold' => true,
                    ],
                    'fill' => [
                        'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                        'startColor' => [
                            'rgb' => 'FFFF00',
                        ],
                    ],
                ]);

                // Apply array of styles to the entire range
                $sheet->getStyle('A1:' . $highestColumn . $highestRow)->applyFromArray([
                    'borders' => [
                        'allBorders' => [
                            'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                            'color' => ['argb' => '000000'],
                        ],
                    ],
                ]);

                // Merge cells for goal spans
                $currentRow = 2; // Start from the second row considering header
                foreach ($this->indicatorCounts as $count) {
                    $goalId = $count->goal_id;
                    $indicatorCount = $count->indicator_count;

                    $rows = $this->data->where('goal_id', $goalId)->keys()->toArray();
                    if (!empty($rows)) {
                        $lastRow = $currentRow + $indicatorCount - 1; // Calculate the last row for the current merge
                        $sheet->mergeCells("B{$currentRow}:B{$lastRow}");
                        $currentRow = $lastRow + 1; // Update the current row to the next row after the last merged row
                    }
                }

                // Merge cells for target spans
                $currentRow = 2; // Start from the second row considering header
                foreach ($this->targetCounts as $count) {
                    $targetId = $count->target_id;
                    $targetCount = $count->indicator_count;

                    $rows = $this->data->where('target_id', $targetId)->keys()->toArray();
                    if (!empty($rows)) {
                        $lastRow = $currentRow + $targetCount - 1; // Calculate the last row for the current merge
                        $sheet->mergeCells("C{$currentRow}:C{$lastRow}");
                        $currentRow = $lastRow + 1; // Update the current row to the next row after the last merged row
                    }
                }
            },
        ];
    }
}
